<?php
/*─────────────────────────────────────────────────────────────────────────────
  SmallMart v6 -  Commercial E-Commerce Platform
  Production-Ready  Marketplace Solution with Enhanced Mod System
  Copyright 2025 SmallMart - Commercial License
  https://smallmart.site
─────────────────────────────────────────────────────────────────────────────*/

declare(strict_types=1);
session_start();
error_reporting(E_ALL & ~E_NOTICE & ~E_WARNING);

// Security Headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: SAMEORIGIN');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Content-Security-Policy: default-src \'self\' \'unsafe-inline\' \'unsafe-eval\' https: data: *.paypal.com *.paypalobjects.com');

// Configuration Constants
define('VERSION', '6.1.5');
define('COPYRIGHT_YEAR', '2026');
define('POWERED_BY_URL', 'https://smallmart.site');
define('DEFAULT_ADMIN_PASSWORD', 'demopass');
define('SHOW_BACKLINK', true);
 // Set to false to permanently remove backlink
define('DATA_DIR', __DIR__ . '/data/');
define('UPLOADS_DIR', __DIR__ . '/uploads/');
define('MODS_DIR', __DIR__ . '/mods/');
define('THEMES_DIR', __DIR__ . '/themes/');
define('UPLOADS_WEB_PATH', 'uploads/');
define('MODS_WEB_PATH', 'mods/');
define('THEMES_WEB_PATH', 'themes/');
define('BACKUP_DIR', DATA_DIR . 'backups/');
define('LOGS_DIR', DATA_DIR . 'logs/');
// Backlink removal license (PayPal IPN verified)
define('BACKLINK_REMOVAL_PRICE', 5.00);
define('BACKLINK_REMOVAL_PAYPAL_EMAIL', 'maddognoland1993@gmail.com');
define('LICENSE_FILE', DATA_DIR . 'license.json');

// Auth / tokens
define('TOKENS_FILE', DATA_DIR . 'tokens.json');
define('SECRET_FILE', DATA_DIR . 'secret.key');
define('EMAIL_VERIFY_TTL_SECONDS', 48 * 3600);
define('PASSWORD_RESET_TTL_SECONDS', 3600);

// Auctions / Offers
define('BIDS_FILE', DATA_DIR . 'bids.json');
define('OFFERS_FILE', DATA_DIR . 'offers.json');

// Logo
define('LOGO_DIR', UPLOADS_DIR . 'branding/');

define('MAX_IMAGES_PER_ITEM', 50);
define('MAX_FILE_SIZE', 10 * 1024 * 1024); // 10MB
define('ITEMS_PER_PAGE', 12);

// Data Files
define('ITEMS_FILE', DATA_DIR . 'items.json');
define('SETTINGS_FILE', DATA_DIR . 'settings.json');
define('PAGES_FILE', DATA_DIR . 'pages.json');
define('ORDERS_FILE', DATA_DIR . 'orders.json');
define('REVIEWS_FILE', DATA_DIR . 'reviews.json');
define('CATEGORIES_FILE', DATA_DIR . 'categories.json');
define('THEMES_FILE', DATA_DIR . 'themes.json');
define('MODS_FILE', DATA_DIR . 'mods.json');
define('STATS_FILE', DATA_DIR . 'stats.json');
define('COUPONS_FILE', DATA_DIR . 'coupons.json');
define('USERS_FILE', DATA_DIR . 'users.json');
define('TRANSACTIONS_FILE', DATA_DIR . 'transactions.json');
define('DONATIONS_FILE', DATA_DIR . 'donations.json');
define('CUSTOM_MODS_FILE', DATA_DIR . 'custom_mods.json');
define('SYSTEM_LOG_FILE', LOGS_DIR . 'system.log');
define('ERROR_LOG_FILE', LOGS_DIR . 'error.log');
define('MOD_LOG_FILE', LOGS_DIR . 'mod.log');
define('ADMIN_LOG_FILE', LOGS_DIR . 'admin.log');

// Initialize directories
foreach ([DATA_DIR, UPLOADS_DIR, MODS_DIR, THEMES_DIR, BACKUP_DIR, LOGS_DIR] as $dir) {
    if (!is_dir($dir)) mkdir($dir, 0755, true);
}

// 10 Pre-built  Themes
$defaultThemes = [
    'professional_dark' => [
        'name' => ' Dark (Default)',
        'category' => 'business',
        'is_default' => true,
        'colors' => [
            'primary' => '#ffcc00',
            'secondary' => '#ff9900',
            'accent' => '#ffd700',
            'background' => 'linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%)',
            'surface' => 'rgba(45, 45, 45, 0.95)',
            'card' => 'rgba(60, 60, 60, 0.9)',
            'text' => '#ffffff',
            'text_secondary' => '#e0e0e0',
            'border' => 'rgba(255, 204, 0, 0.2)'
        ],
        'fonts' => ['primary' => "'Inter', sans-serif", 'heading' => "'Poppins', sans-serif"]
    ],
    'ocean_blue' => [
        'name' => 'Ocean Blue',
        'category' => 'modern',
        'is_default' => true,
        'colors' => [
            'primary' => '#0ea5e9',
            'secondary' => '#0284c7',
            'accent' => '#38bdf8',
            'background' => 'linear-gradient(135deg, #0f172a 0%, #1e293b 100%)',
            'surface' => 'rgba(30, 41, 59, 0.95)',
            'card' => 'rgba(51, 65, 85, 0.9)',
            'text' => '#f8fafc',
            'text_secondary' => '#cbd5e1',
            'border' => 'rgba(14, 165, 233, 0.2)'
        ],
        'fonts' => ['primary' => "'Roboto', sans-serif", 'heading' => "'Montserrat', sans-serif"]
    ],
    'forest_green' => [
        'name' => 'Forest Green',
        'category' => 'nature',
        'is_default' => true,
        'colors' => [
            'primary' => '#22c55e',
            'secondary' => '#16a34a',
            'accent' => '#4ade80',
            'background' => 'linear-gradient(135deg, #064e3b 0%, #065f46 100%)',
            'surface' => 'rgba(6, 95, 70, 0.95)',
            'card' => 'rgba(20, 83, 45, 0.9)',
            'text' => '#ecfdf5',
            'text_secondary' => '#bbf7d0',
            'border' => 'rgba(34, 197, 94, 0.2)'
        ],
        'fonts' => ['primary' => "'Open Sans', sans-serif", 'heading' => "'Nunito', sans-serif"]
    ],
    'sunset_orange' => [
        'name' => 'Sunset Orange',
        'category' => 'vibrant',
        'is_default' => true,
        'colors' => [
            'primary' => '#f97316',
            'secondary' => '#ea580c',
            'accent' => '#fb923c',
            'background' => 'linear-gradient(135deg, #431407 0%, #7c2d12 100%)',
            'surface' => 'rgba(124, 45, 18, 0.95)',
            'card' => 'rgba(154, 52, 18, 0.9)',
            'text' => '#fff7ed',
            'text_secondary' => '#fed7aa',
            'border' => 'rgba(249, 115, 22, 0.2)'
        ],
        'fonts' => ['primary' => "'Lato', sans-serif", 'heading' => "'Playfair Display', serif"]
    ],
    'royal_purple' => [
        'name' => 'Royal Purple',
        'category' => 'luxury',
        'is_default' => true,
        'colors' => [
            'primary' => '#8b5cf6',
            'secondary' => '#7c3aed',
            'accent' => '#a78bfa',
            'background' => 'linear-gradient(135deg, #2e1065 0%, #4c1d95 100%)',
            'surface' => 'rgba(76, 29, 149, 0.95)',
            'card' => 'rgba(91, 33, 182, 0.9)',
            'text' => '#faf5ff',
            'text_secondary' => '#ddd6fe',
            'border' => 'rgba(139, 92, 246, 0.2)'
        ],
        'fonts' => ['primary' => "'Source Sans Pro', sans-serif", 'heading' => "'Merriweather', serif"]
    ],
    'crimson_red' => [
        'name' => 'Crimson Red',
        'category' => 'bold',
        'is_default' => true,
        'colors' => [
            'primary' => '#dc2626',
            'secondary' => '#b91c1c',
            'accent' => '#ef4444',
            'background' => 'linear-gradient(135deg, #450a0a 0%, #7f1d1d 100%)',
            'surface' => 'rgba(127, 29, 29, 0.95)',
            'card' => 'rgba(153, 27, 27, 0.9)',
            'text' => '#fef2f2',
            'text_secondary' => '#fecaca',
            'border' => 'rgba(220, 38, 38, 0.2)'
        ],
        'fonts' => ['primary' => "'Roboto', sans-serif", 'heading' => "'Oswald', sans-serif"]
    ],
    'minimalist_white' => [
        'name' => 'Minimalist White',
        'category' => 'clean',
        'is_default' => true,
        'colors' => [
            'primary' => '#374151',
            'secondary' => '#1f2937',
            'accent' => '#6b7280',
            'background' => 'linear-gradient(135deg, #ffffff 0%, #f9fafb 100%)',
            'surface' => 'rgba(255, 255, 255, 0.95)',
            'card' => 'rgba(249, 250, 251, 0.9)',
            'text' => '#111827',
            'text_secondary' => '#6b7280',
            'border' => 'rgba(55, 65, 81, 0.1)'
        ],
        'fonts' => ['primary' => "'Inter', sans-serif", 'heading' => "'Inter', sans-serif"]
    ],
    'cyberpunk_neon' => [
        'name' => 'Cyberpunk Neon',
        'category' => 'futuristic',
        'is_default' => true,
        'colors' => [
            'primary' => '#00ff9f',
            'secondary' => '#ff0080',
            'accent' => '#00d4ff',
            'background' => 'linear-gradient(135deg, #0a0a0a 0%, #1a0a1a 100%)',
            'surface' => 'rgba(26, 10, 26, 0.95)',
            'card' => 'rgba(40, 20, 40, 0.9)',
            'text' => '#ffffff',
            'text_secondary' => '#b3b3b3',
            'border' => 'rgba(0, 255, 159, 0.3)'
        ],
        'fonts' => ['primary' => "'Orbitron', monospace", 'heading' => "'Orbitron', monospace"]
    ],
    'autumn_warmth' => [
        'name' => 'Autumn Warmth',
        'category' => 'seasonal',
        'is_default' => true,
        'colors' => [
            'primary' => '#d97706',
            'secondary' => '#b45309',
            'accent' => '#f59e0b',
            'background' => 'linear-gradient(135deg, #451a03 0%, #78350f 100%)',
            'surface' => 'rgba(120, 53, 15, 0.95)',
            'card' => 'rgba(146, 64, 14, 0.9)',
            'text' => '#fef3c7',
            'text_secondary' => '#fcd34d',
            'border' => 'rgba(217, 119, 6, 0.2)'
        ],
        'fonts' => ['primary' => "'Georgia', serif", 'heading' => "'Crimson Text', serif"]
    ],
    'arctic_ice' => [
        'name' => 'Arctic Ice',
        'category' => 'cool',
        'is_default' => true,
        'colors' => [
            'primary' => '#06b6d4',
            'secondary' => '#0891b2',
            'accent' => '#67e8f9',
            'background' => 'linear-gradient(135deg, #083344 0%, #155e75 100%)',
            'surface' => 'rgba(21, 94, 117, 0.95)',
            'card' => 'rgba(8, 51, 68, 0.9)',
            'text' => '#f0f9ff',
            'text_secondary' => '#bae6fd',
            'border' => 'rgba(6, 182, 212, 0.2)'
        ],
        'fonts' => ['primary' => "'Nunito Sans', sans-serif", 'heading' => "'Raleway', sans-serif"]
    ]
];

// Premium Categories with Icons
$premiumCategories = [
    'electronics' => ['name' => 'Electronics & Technology', 'icon' => 'fas fa-laptop', 'enabled' => true],
    'fashion' => ['name' => 'Fashion & Clothing', 'icon' => 'fas fa-tshirt', 'enabled' => true],
    'home_garden' => ['name' => 'Home & Garden', 'icon' => 'fas fa-home', 'enabled' => true],
    'sports_outdoors' => ['name' => 'Sports & Outdoors', 'icon' => 'fas fa-football-ball', 'enabled' => true],
    'books_media' => ['name' => 'Books & Media', 'icon' => 'fas fa-book', 'enabled' => true],
    'toys_games' => ['name' => 'Toys & Games', 'icon' => 'fas fa-gamepad', 'enabled' => true],
    'automotive' => ['name' => 'Automotive & Motors', 'icon' => 'fas fa-car', 'enabled' => true],
    'health_beauty' => ['name' => 'Health & Beauty', 'icon' => 'fas fa-heart', 'enabled' => true],
    'collectibles' => ['name' => 'Collectibles & Art', 'icon' => 'fas fa-gem', 'enabled' => true],
    'business_industrial' => ['name' => 'Business & Industrial', 'icon' => 'fas fa-industry', 'enabled' => false],
    'pets_supplies' => ['name' => 'Pet Supplies', 'icon' => 'fas fa-paw', 'enabled' => false],
    'jewelry_watches' => ['name' => 'Jewelry & Watches', 'icon' => 'fas fa-ring', 'enabled' => false]
];

// Enhanced Built-in Mod System with 15  Mods
$builtinMods = [
    'advanced_search' => [
        'name' => 'Advanced Search & Filters Pro',
        'description' => ' search with price ranges, multiple filters, saved searches, and smart suggestions',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['search_form', 'search_results', 'item_filters', 'search_before', 'search_after'],
        'api_version' => '1.0',
        'settings' => [
            'enable_price_filter' => true,
            'enable_category_filter' => true,
            'enable_condition_filter' => true,
            'enable_location_filter' => true,
            'enable_saved_searches' => true,
            'max_price_range' => 50000
        ]
    ],
    'theme_builder' => [
        'name' => 'Custom Theme Builder Pro',
        'description' => ' theme builder with live preview, color picker, export/import themes',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['admin_menu', 'theme_settings', 'theme_export', 'theme_import'],
        'api_version' => '1.0',
        'settings' => [
            'enable_live_preview' => true,
            'enable_color_picker' => true,
            'enable_theme_export' => true,
            'enable_theme_import' => true,
            'max_custom_themes' => 10
        ]
    ],
    'paypal_integration' => [
        'name' => 'PayPal Payment Gateway',
        'description' => 'Simple PayPal integration using email address for payments',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['payment_methods', 'order_processing', 'payment_form', 'payment_validation'],
        'api_version' => '1.0',
        'settings' => [
            'paypal_email' => '',
            'sandbox_mode' => false
        ]
    ],
    'social_sharing' => [
        'name' => 'Social Media Integration Suite',
        'description' => 'Complete social sharing with 10+ platforms and analytics',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['item_view', 'social_buttons', 'share_tracking', 'social_analytics'],
        'api_version' => '1.0',
        'settings' => [
            'platforms' => ['facebook', 'twitter', 'instagram', 'pinterest', 'whatsapp', 'linkedin'],
            'track_shares' => true
        ]
    ],
    'analytics_pro' => [
        'name' => ' Analytics Dashboard',
        'description' => 'Enterprise-grade analytics with real-time tracking and detailed reports',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['page_view', 'analytics_dashboard', 'tracking_script', 'analytics_report'],
        'api_version' => '1.0',
        'settings' => [
            'real_time_tracking' => true,
            'conversion_tracking' => true,
            'retention_days' => 365
        ]
    ],
    'seo_optimizer' => [
        'name' => 'SEO Optimization Engine',
        'description' => 'Automatic SEO optimization with meta tags, sitemaps, schema markup, and keyword analysis',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['head_meta', 'sitemap_generation', 'schema_markup', 'seo_analysis'],
        'api_version' => '1.0',
        'settings' => [
            'auto_meta_tags' => true,
            'generate_sitemap' => true,
            'schema_markup' => true,
            'keyword_analysis' => true,
            'google_analytics_id' => ''
        ]
    ],
    'image_optimizer' => [
        'name' => 'Image Compression & Optimization',
        'description' => 'Automatic image compression, resizing, WebP conversion, and lazy loading',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['image_upload', 'image_display', 'lazy_loading'],
        'api_version' => '1.0',
        'settings' => [
            'auto_compress' => true,
            'webp_conversion' => true,
            'lazy_loading' => true,
            'max_width' => 1200,
            'quality' => 85
        ]
    ],
    'inventory_manager' => [
        'name' => 'Advanced Inventory Management',
        'description' => 'Track stock levels, low stock alerts, bulk operations, and inventory reports',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['stock_update', 'low_stock_alert', 'inventory_report'],
        'api_version' => '1.0',
        'settings' => [
            'track_inventory' => true,
            'low_stock_threshold' => 5,
            'auto_hide_zero_stock' => false,
            'email_low_stock_alerts' => false
        ]
    ],
    'coupon_system' => [
        'name' => 'Discount & Coupon System',
        'description' => 'Create percentage, fixed amount, and bulk discounts with expiration dates',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['checkout_process', 'coupon_validation', 'discount_calculation'],
        'api_version' => '1.0',
        'settings' => [
            'allow_coupon_stacking' => false,
            'minimum_order_amount' => 0,
            'max_discount_percent' => 50,
            'auto_apply_best_discount' => true
        ]
    ],
    'security_scanner' => [
        'name' => 'Security Scanner & Monitor',
        'description' => 'Monitor for security threats, file changes, and suspicious activity',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['security_scan', 'file_integrity', 'login_monitoring'],
        'api_version' => '1.0',
        'settings' => [
            'scan_frequency' => 'daily',
            'monitor_file_changes' => true,
            'log_failed_logins' => true,
            'email_security_alerts' => true
        ]
    ],
    'advanced_statistics' => [
        'name' => 'Advanced Statistics & Reports',
        'description' => 'Detailed sales reports, customer analytics, and business intelligence dashboard',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['statistics_calculation', 'report_generation', 'dashboard_widgets'],
        'api_version' => '1.0',
        'settings' => [
            'generate_daily_reports' => true,
            'track_customer_behavior' => true,
            'sales_forecasting' => true,
            'export_reports' => true
        ]
    ],
    'product_comparison' => [
        'name' => 'Product Comparison Tool',
        'description' => 'Allow customers to compare multiple products side-by-side with detailed feature comparison',
        'version' => '6.1.5',
        'author' => 'SmallMart',
        'enabled' => true,
        'status' => 'active',
        'type' => 'core',
        'hooks' => ['comparison_widget', 'product_features', 'comparison_table'],
        'api_version' => '1.0',
        'settings' => [
            'max_comparison_items' => 4,
            'show_comparison_button' => true,
            'highlight_differences' => true,
            'save_comparisons' => true
        ]
    ]
];

// Mod API Class
class SmallMartModAPI {
    private static array $hooks = [];
    private static array $settings = [];
    
    public static function addHook(string $hookName, callable $callback, int $priority = 10): void {
        if (!isset(self::$hooks[$hookName])) {
            self::$hooks[$hookName] = [];
        }
        self::$hooks[$hookName][] = ['callback' => $callback, 'priority' => $priority];
    }
    
    public static function executeHook(string $hookName, $data = null) {
        if (!isset(self::$hooks[$hookName])) return $data;
        
        // Sort by priority
        usort(self::$hooks[$hookName], function($a, $b) {
            return $a['priority'] <=> $b['priority'];
        });
        
        foreach (self::$hooks[$hookName] as $hook) {
            if (is_callable($hook['callback'])) {
                $data = call_user_func($hook['callback'], $data);
            }
        }
        
        return $data;
    }
    
    public static function getSetting(string $key, $default = null) {
        return self::$settings[$key] ?? $default;
    }
    
    public static function setSetting(string $key, $value): void {
        self::$settings[$key] = $value;
    }
    
    public static function log(string $message, string $level = 'info'): void {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] [{$level}] {$message}\n";
        @file_put_contents(MOD_LOG_FILE, $logEntry, FILE_APPEND | LOCK_EX);
    }
}

// Enhanced Logging System
class Logger {
    public static function system(string $message): void {
        self::writeLog(SYSTEM_LOG_FILE, $message, 'SYSTEM');
    }
    
    public static function error(string $message): void {
        self::writeLog(ERROR_LOG_FILE, $message, 'ERROR');
    }
    
    public static function admin(string $message): void {
        self::writeLog(ADMIN_LOG_FILE, $message, 'ADMIN');
    }
    
    public static function mod(string $message): void {
        self::writeLog(MOD_LOG_FILE, $message, 'MOD');
    }
    
    private static function writeLog(string $file, string $message, string $type): void {
        $timestamp = date('Y-m-d H:i:s');
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user = isAdmin() ? 'admin' : 'guest';
        $logEntry = "[{$timestamp}] [{$type}] [{$user}@{$ip}] {$message}\n";
        @file_put_contents($file, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    public static function readLog(string $file, int $lines = 50): array {
        if (!file_exists($file)) return [];
        
        $content = @file_get_contents($file);
        if ($content === false) return [];
        
        $logLines = array_filter(explode("\n", $content));
        return array_slice(array_reverse($logLines), 0, $lines);
    }
    
    public static function clearLog(string $file): bool {
        return @file_put_contents($file, '') !== false;
    }
}

// Utility Functions
function sanitizeInput($input, bool $allowHtml = false): string {
    if ($input === null || $input === '' || $input === false) return '';
    
    $input = (string) $input;
    $input = trim($input);
    
    if ($allowHtml) {
        $allowedTags = '<p><br><strong><em><u><ol><ul><li><h1><h2><h3><h4><h5><h6><a><img><blockquote><code>';
        $input = strip_tags($input, $allowedTags);
        return $input;
    } else {
        $input = strip_tags($input);
        return htmlspecialchars($input, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8', false);
    }
}

function safeGet(array $array, string $key, $default = '') {
    return isset($array[$key]) && $array[$key] !== null ? $array[$key] : $default;
}

function generateCSRF(): string {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function validateCSRF(?string $token): bool {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token ?? '');
}

function generateId(string $prefix = 'item'): string {
    return $prefix . '_' . uniqid() . '_' . time();
}

function formatPrice($price, string $currency = '$'): string {
    if ($price === null || $price === '' || $price === false) return $currency . '0.00';
    if (!is_numeric($price)) return sanitizeInput((string)$price, false);
    return $currency . number_format(floatval($price), 2);
}

function timeAgo($datetime): string {
    if (!$datetime || empty($datetime)) return 'Unknown';
    
    $timestamp = is_numeric($datetime) ? $datetime : strtotime($datetime);
    if ($timestamp === false || $timestamp <= 0) return 'Unknown';
    
    $time = time() - $timestamp;
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . 'm ago';
    if ($time < 86400) return floor($time/3600) . 'h ago';
    if ($time < 2592000) return floor($time/86400) . 'd ago';
    return date('M j, Y', $timestamp);
}

function loadData(string $file): array {
    if (!file_exists($file)) return [];
    
    $content = @file_get_contents($file);
    if ($content === false) return [];
    
    $data = @json_decode($content, true);
    return is_array($data) ? $data : [];
}

function saveData(string $file, array $data): bool {
    if (!is_array($data)) return false;
    
    $json = @json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    if ($json === false) return false;
    
    return @file_put_contents($file, $json, LOCK_EX) !== false;
}

function isAdmin(): bool {
    return isset($_SESSION['admin']) && $_SESSION['admin'] === true;
}

function requireAdmin(): void {
    if (!isAdmin()) {
        http_response_code(403);
        exit('Access denied');
    }
}

function createBackup(): bool {
    try {
        $timestamp = date('Y-m-d_H-i-s');
        $backupFile = BACKUP_DIR . "backup_$timestamp.json";
        $backupData = [
            'timestamp' => $timestamp,
            'version' => VERSION,
            'items' => loadData(ITEMS_FILE),
            'settings' => loadData(SETTINGS_FILE),
            'orders' => loadData(ORDERS_FILE),
            'reviews' => loadData(REVIEWS_FILE),
            'categories' => loadData(CATEGORIES_FILE),
            'themes' => loadData(THEMES_FILE),
            'mods' => loadData(MODS_FILE),
            'custom_mods' => loadData(CUSTOM_MODS_FILE),
            'stats' => loadData(STATS_FILE),
            'users' => loadData(USERS_FILE),
            'transactions' => loadData(TRANSACTIONS_FILE),
            'donations' => loadData(DONATIONS_FILE)
        ];
        Logger::admin("Backup created: $backupFile");
        return saveData($backupFile, $backupData);
    } catch (Exception $e) {
        Logger::error("Backup failed: " . $e->getMessage());
        return false;
    }
}

function generateThemeCSS(array $themeData): string {
    if (!is_array($themeData) || !isset($themeData['colors']) || !is_array($themeData['colors'])) {
        return '';
    }
    
    $colors = $themeData['colors'];
    $fonts = isset($themeData['fonts']) && is_array($themeData['fonts']) ? $themeData['fonts'] : [];
    
    $defaultColors = [
        'primary' => '#ffcc00',
        'secondary' => '#ff9900', 
        'accent' => '#ffd700',
        'background' => 'linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%)',
        'surface' => 'rgba(45, 45, 45, 0.95)',
        'card' => 'rgba(60, 60, 60, 0.9)',
        'text' => '#ffffff',
        'text_secondary' => '#e0e0e0',
        'border' => 'rgba(255, 204, 0, 0.2)'
    ];
    
    foreach ($defaultColors as $key => $default) {
        if (!isset($colors[$key]) || empty($colors[$key])) {
            $colors[$key] = $default;
        }
    }
    
    return "
        :root {
            --primary: {$colors['primary']};
            --secondary: {$colors['secondary']};
            --accent: {$colors['accent']};
            --background: {$colors['background']};
            --surface: {$colors['surface']};
            --card: {$colors['card']};
            --text: {$colors['text']};
            --text-secondary: {$colors['text_secondary']};
            --border: {$colors['border']};
            --font-primary: " . (isset($fonts['primary']) && !empty($fonts['primary']) ? $fonts['primary'] : "'Inter', sans-serif") . ";
            --font-heading: " . (isset($fonts['heading']) && !empty($fonts['heading']) ? $fonts['heading'] : "'Poppins', sans-serif") . ";
        }";
}

// Replace the existing mod-related functions (around lines 580-650) with these improved versions:

// Enhanced mod metadata extraction
function extractModMetadata(string $filePath): array {
    $metadata = [
        'name' => '',
        'description' => '',
        'version' => '1.0.0',
        'author' => 'Unknown',
        'enabled' => true,
        'status' => 'active',
        'type' => 'custom',
        'hooks' => [],
        'settings' => []
    ];
    
    $content = @file_get_contents($filePath);
    if ($content === false) return $metadata;
    
    // Extract header comment block
    if (preg_match('/\/\*\s*(.*?)\*\//s', $content, $matches)) {
        $header = $matches[1];
        
        // Extract required fields
        if (preg_match('/SmallMart Mod:\s*(.+)/i', $header, $match)) {
            $metadata['name'] = trim($match[1]);
        }
        if (preg_match('/Description:\s*(.+)/i', $header, $match)) {
            $metadata['description'] = trim($match[1]);
        }
        if (preg_match('/Version:\s*(.+)/i', $header, $match)) {
            $metadata['version'] = trim($match[1]);
        }
        if (preg_match('/Author:\s*(.+)/i', $header, $match)) {
            $metadata['author'] = trim($match[1]);
        }
        
        // Extract hooks by looking for addHook calls
        if (preg_match_all('/addHook\([\'"]([^\'"]+)[\'"]/i', $content, $hookMatches)) {
            $metadata['hooks'] = array_unique($hookMatches[1]);
        }
    }
    
    return $metadata;
}

// Improved validation
function validateModFile(string $filePath): array {
    $errors = [];
    
    if (!file_exists($filePath)) {
        $errors[] = 'Mod file does not exist';
        return $errors;
    }
    
    $content = @file_get_contents($filePath);
    if ($content === false) {
        $errors[] = 'Unable to read mod file';
        return $errors;
    }
    
    // Check for PHP opening tag
    if (strpos($content, '<?php') === false) {
        $errors[] = 'Mod file must be a valid PHP file';
    }
    
    // Check for required mod header
    if (strpos($content, 'SmallMart Mod:') === false) {
        $errors[] = 'Mod file missing required header comment with "SmallMart Mod:" field';
    }
    
    // Check for dangerous functions in unsafe contexts (more nuanced check)
    $dangerousFunctions = ['exec', 'system', 'shell_exec', 'passthru', 'eval'];
    foreach ($dangerousFunctions as $func) {
        if (preg_match('/\b' . $func . '\s*\(/i', $content)) {
            $errors[] = "Dangerous function '{$func}' detected - not allowed";
        }
    }
    
    // Check for direct file system modifications outside approved directories
    if (preg_match('/unlink\s*\([\'"](?!.*\/mods\/|.*\/uploads\/)/i', $content)) {
        $errors[] = "Unauthorized file deletion detected";
    }
    
    return $errors;
}

// Fixed installation function
function installCustomMod(string $filePath, string $modId): bool {
    $errors = validateModFile($filePath);
    if (!empty($errors)) {
        Logger::error('Mod validation failed: ' . implode(', ', $errors));
        return false;
    }
    
    $targetPath = MODS_DIR . $modId . '.php';
    if (!@copy($filePath, $targetPath)) {
        Logger::error('Failed to copy mod file to: ' . $targetPath);
        return false;
    }
    
    // Extract metadata and register the mod
    $metadata = extractModMetadata($targetPath);
    if (empty($metadata['name'])) {
        $metadata['name'] = ucfirst(str_replace('_', ' ', $modId));
    }
    
    // Save to custom mods registry
    $customMods = loadData(CUSTOM_MODS_FILE);
    $customMods[$modId] = $metadata;
    saveData(CUSTOM_MODS_FILE, $customMods);
    
    // Also add to main mods array
    $mods = loadData(MODS_FILE);
    $mods[$modId] = $metadata;
    saveData(MODS_FILE, $mods);
    
    Logger::admin('Custom mod installed and registered: ' . $modId);
    return true;
}

// Fixed upload handler
function handleModUpload(): array {
    $result = ['success' => false, 'message' => '', 'mod_id' => ''];
    
    if (!isset($_FILES['mod_file']) || $_FILES['mod_file']['error'] !== UPLOAD_ERR_OK) {
        $result['message'] = 'No mod file uploaded or upload error';
        return $result;
    }
    
    $allowedTypes = ['application/x-php', 'text/x-php', 'text/plain', 'application/octet-stream'];
    $fileType = $_FILES['mod_file']['type'];
    $fileName = $_FILES['mod_file']['name'];
    
    if (!in_array($fileType, $allowedTypes) && pathinfo($fileName, PATHINFO_EXTENSION) !== 'php') {
        $result['message'] = 'Invalid file type. Only PHP files are allowed.';
        return $result;
    }
    
    if ($_FILES['mod_file']['size'] > 1024 * 1024) {
        $result['message'] = 'File too large. Maximum size is 1MB.';
        return $result;
    }
    
    // Generate mod ID from filename or use provided one
    $modId = sanitizeInput($_POST['mod_id'] ?? pathinfo($fileName, PATHINFO_FILENAME));
    $modId = preg_replace('/[^a-z0-9_]/i', '_', $modId);
    
    if (empty($modId)) {
        $result['message'] = 'Invalid mod ID';
        return $result;
    }
    
    if (installCustomMod($_FILES['mod_file']['tmp_name'], $modId)) {
        $result['success'] = true;
        $result['message'] = 'Mod uploaded and installed successfully! Refresh the page to see it in action.';
        $result['mod_id'] = $modId;
    } else {
        $result['message'] = 'Failed to install mod. Check the error log for details.';
    }
    
    return $result;
}

// Fixed loader function
function loadCustomMods(): array {
    $customMods = loadData(CUSTOM_MODS_FILE);
    $mods = loadData(MODS_FILE);
    if (!is_array($customMods)) $customMods = [];
    if (!is_array($mods)) $mods = [];

    // Legacy single-file mods (/mods/*.php)
    $modFiles = glob(MODS_DIR . '*.php') ?: [];
    foreach ($modFiles as $modFile) {
        $modId = basename($modFile, '.php');
        if (isset($mods[$modId]) && !($mods[$modId]['enabled'] ?? true)) continue;

        try {
            include_once $modFile;
            $safeId = preg_replace('/[^a-zA-Z0-9_]/', '_', $modId);
            $initFunctions = [$safeId . '_init', 'init', $safeId . '_initialize'];
            foreach ($initFunctions as $initFunc) {
                if (function_exists($initFunc)) { call_user_func($initFunc); break; }
            }
            if (!isset($customMods[$modId])) {
                $metadata = extractModMetadata($modFile);
                $customMods[$modId] = $metadata;
                if (!isset($mods[$modId])) $mods[$modId] = $metadata;
            }
            Logger::mod("Custom mod loaded: $modId");
        } catch (Exception $e) {
            Logger::error("Failed to load custom mod $modId: " . $e->getMessage());
        }
    }

    // Package mods (/mods/<id>/manifest.json)
    $packageDirs = glob(MODS_DIR . '*', GLOB_ONLYDIR) ?: [];
    foreach ($packageDirs as $dir) {
        $manifestPath = rtrim($dir, "/\\") . DIRECTORY_SEPARATOR . 'manifest.json';
        if (!file_exists($manifestPath)) continue;

        $manifest = json_decode(@file_get_contents($manifestPath), true);
        if (!is_array($manifest)) continue;

        $modId = $manifest['id'] ?? basename($dir);
        $enabled = $mods[$modId]['enabled'] ?? ($manifest['enabled_by_default'] ?? true);
        if (!$enabled) continue;

        $entry = $manifest['entry'] ?? 'main.php';
        $entryPath = rtrim($dir, "/\\") . DIRECTORY_SEPARATOR . $entry;
        if (!file_exists($entryPath)) {
            Logger::error("Mod package missing entry: $modId");
            continue;
        }

        try {
            include_once $entryPath;
            $safeId = preg_replace('/[^a-zA-Z0-9_]/', '_', $modId);
            $initFunctions = [$safeId . '_init', 'init', $safeId . '_initialize'];
            foreach ($initFunctions as $initFunc) {
                if (function_exists($initFunc)) { call_user_func($initFunc); break; }
            }

            $meta = [
                'id' => $modId,
                'name' => $manifest['name'] ?? $modId,
                'version' => $manifest['version'] ?? '1.0.0',
                'description' => $manifest['description'] ?? '',
                'author' => $manifest['author'] ?? '',
                'enabled' => true,
                'type' => 'package',
                'dir' => basename($dir),
            ];
            $customMods[$modId] = $meta;
            if (!isset($mods[$modId])) $mods[$modId] = $meta;
            Logger::mod("Mod package loaded: $modId");
        } catch (Exception $e) {
            Logger::error("Failed to load mod package $modId: " . $e->getMessage());
        }
    }

    saveData(CUSTOM_MODS_FILE, $customMods);
    saveData(MODS_FILE, $mods);
    return $customMods;
}

function exportTheme(string $themeId): string {
    $themes = loadData(THEMES_FILE);
    // Auto-discover folder themes (/themes/<id>/theme.json)
    $themeDirs = glob(THEMES_DIR . '*', GLOB_ONLYDIR) ?: [];
    foreach ($themeDirs as $td) {
        $tp = rtrim($td, "/\\") . DIRECTORY_SEPARATOR . 'theme.json';
        if (!file_exists($tp)) continue;
        $tj = json_decode(@file_get_contents($tp), true);
        if (!is_array($tj)) continue;
        $tid = $tj['id'] ?? basename($td);
        if (!isset($themes[$tid])) { $themes[$tid] = $tj; }
    }
    if (!isset($themes[$themeId])) {
        return '';
    }
    
    $theme = $themes[$themeId];
    $theme['export_version'] = VERSION;
    $theme['export_date'] = date('Y-m-d H:i:s');
    $theme['export_id'] = $themeId;
    
    return base64_encode(json_encode($theme, JSON_PRETTY_PRINT));
}

function importTheme(string $themeData): array {
    $result = ['success' => false, 'message' => '', 'theme_id' => 'smallmart-yellow'];
    
    $decoded = @base64_decode($themeData);
    if ($decoded === false) {
        $result['message'] = 'Invalid theme data format';
        return $result;
    }
    
    $theme = @json_decode($decoded, true);
    if (!is_array($theme)) {
        $result['message'] = 'Invalid theme JSON structure';
        return $result;
    }
    
    // Validate required theme structure
    if (!isset($theme['name']) || !isset($theme['colors']) || !is_array($theme['colors'])) {
        $result['message'] = 'Theme missing required structure (name, colors)';
        return $result;
    }
    
    $themes = loadData(THEMES_FILE);
    $themeId = 'imported_' . time() . '_' . substr(md5($theme['name']), 0, 8);
    
    // Mark as imported and custom
    $theme['is_custom'] = true;
    $theme['is_imported'] = true;
    $theme['imported_date'] = date('Y-m-d H:i:s');
    
    // Remove export metadata
    unset($theme['export_version'], $theme['export_date'], $theme['export_id']);
    
    $themes[$themeId] = $theme;
    
    if (saveData(THEMES_FILE, $themes)) {
        Logger::admin("Theme imported: $themeId");
        $result['success'] = true;
        $result['message'] = 'Theme imported successfully';
        $result['theme_id'] = $themeId;
    } else {
        $result['message'] = 'Failed to save imported theme';
    }
    
    return $result;
}

// PayPal Functions
function generatePayPalForm(array $item, array $settings): string {
    if (!is_array($item) || !is_array($settings)) return '';
    
    $itemPaypalEnabled = $item['enable_paypal'] ?? true;
    $globalPaypalEnabled = $settings['payment_methods']['paypal'] ?? false;
    
    if (!$itemPaypalEnabled || !$globalPaypalEnabled) return '';
    
    $paypalEmail = '';
    if (isset($settings['mods']['paypal_integration']['settings']['paypal_email'])) {
        $paypalEmail = $settings['mods']['paypal_integration']['settings']['paypal_email'];
    }
    
    if (empty($paypalEmail) || !filter_var($paypalEmail, FILTER_VALIDATE_EMAIL)) return '';
    
    $sandbox = isset($settings['mods']['paypal_integration']['settings']['sandbox_mode']) 
        ? $settings['mods']['paypal_integration']['settings']['sandbox_mode'] : false;
    $action = $sandbox ? 'https://www.sandbox.paypal.com/cgi-bin/webscr' : 'https://www.paypal.com/cgi-bin/webscr';
    
    $itemId = isset($item['id']) ? $item['id'] : 'unknown';
    $itemTitle = isset($item['title']) ? sanitizeInput($item['title'], false) : 'Item';
    $itemPrice = isset($item['price']) && is_numeric($item['price']) ? number_format(floatval($item['price']), 2, '.', '') : '0.00';
    
    $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
    $host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost';
    $script = isset($_SERVER['PHP_SELF']) ? $_SERVER['PHP_SELF'] : '/';
    
    $returnUrl = $protocol . '://' . $host . $script . '?paypal_success=1&item_id=' . urlencode($itemId);
    $cancelUrl = $protocol . '://' . $host . $script . '?view=' . urlencode($itemId);
    
    return '
    <form action="' . htmlspecialchars($action) . '" method="post" target="_top">
        <input type="hidden" name="cmd" value="_xclick">
        <input type="hidden" name="business" value="' . htmlspecialchars($paypalEmail) . '">
        <input type="hidden" name="item_name" value="' . htmlspecialchars($itemTitle) . '">
        <input type="hidden" name="amount" value="' . htmlspecialchars($itemPrice) . '">
        <input type="hidden" name="currency_code" value="USD">
        <input type="hidden" name="return" value="' . htmlspecialchars($returnUrl) . '">
        <input type="hidden" name="cancel_return" value="' . htmlspecialchars($cancelUrl) . '">
        <input type="hidden" name="notify_url" value="' . htmlspecialchars($returnUrl) . '">
        <input type="hidden" name="custom" value="' . htmlspecialchars($itemId) . '">
        <button type="submit" class="btn btn-lg" style="width: 100%; background: #0070ba; color: white; padding: 15px; margin-bottom: 10px;">
            <i class="fab fa-paypal"></i> Pay with PayPal - $' . htmlspecialchars($itemPrice) . '
        </button>
    </form>';
}

function isContactSellerEnabled(array $item, array $settings): bool {
    $itemContactEnabled = $item['enable_contact'] ?? true;
    $globalContactEnabled = $settings['payment_methods']['contact'] ?? false;
    return $itemContactEnabled && $globalContactEnabled;
}

// Donation System Functions
function isDonationComplete(): bool {
    // Backwards-compatible name: reflects whether the backlink-removal license is paid.
    $lic = loadData(LICENSE_FILE);
    return is_array($lic) && (($lic['paid'] ?? false) === true);
}

function recordDonation(bool $wasSkipped = false): bool {
    // Backwards-compatible name. Skip is ignored (license must be paid via PayPal IPN).
    $lic = loadData(LICENSE_FILE);
    if (!is_array($lic)) $lic = [];
    $lic['paid'] = true;
    $lic['paid_at'] = date('c');
    $lic['provider'] = 'manual';
    $lic['note'] = 'Manually marked paid (legacy)';
    return saveData(LICENSE_FILE, $lic);
}

function shouldShowBacklink(): bool {
    if (!SHOW_BACKLINK) return false;
    if (!isDonationComplete()) return true;
    $settings = loadData(SETTINGS_FILE);
    if (is_array($settings) && array_key_exists('backlink_show', $settings)) return (bool)$settings['backlink_show'];
    return true;
}


// License + Payments + Logo helpers (v6)
function getSecretKey(): string {
    if (!file_exists(SECRET_FILE)) { @file_put_contents(SECRET_FILE, bin2hex(random_bytes(32))); }
    $k = @file_get_contents(SECRET_FILE);
    $k = is_string($k) ? trim($k) : '';
    if ($k === '') { $k = bin2hex(random_bytes(32)); @file_put_contents(SECRET_FILE, $k); }
    return $k;
}

function baseUrl(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $proto = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $script = $_SERVER['PHP_SELF'] ?? '/';
    return $proto . '://' . $host . $script;
}

function paypalCustomForLicense(): string {
    $site = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $sig = hash_hmac('sha256', 'SM_BACKLINK_REMOVAL|' . $site, getSecretKey());
    return 'SM_BACKLINK_REMOVAL:' . $site . ':' . $sig;
}
function verifyPaypalLicenseCustom(string $custom): bool {
    $parts = explode(':', $custom);
    if (count($parts) !== 3) return false;
    if ($parts[0] !== 'SM_BACKLINK_REMOVAL') return false;
    $site = $parts[1]; $sig = $parts[2];
    $expected = hash_hmac('sha256', 'SM_BACKLINK_REMOVAL|' . $site, getSecretKey());
    return hash_equals($expected, $sig);
}

function paypalVerifyIpn(array $post, bool $sandbox): bool {
    $endpoint = $sandbox ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr' : 'https://ipnpb.paypal.com/cgi-bin/webscr';
    $post['cmd'] = '_notify-validate';
    $ch = curl_init($endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Connection: Close']);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($res === false) { Logger::error('PayPal IPN verify error: ' . $err); return false; }
    return trim((string)$res) === 'VERIFIED';
}

function handlePaypalLicenseIpn(array $settings): void {
    $post = $_POST;
    $sandbox = (bool)($settings['paypal_sandbox'] ?? false);
    if (!paypalVerifyIpn($post, $sandbox)) { http_response_code(400); echo 'INVALID'; exit; }
    $paymentStatus = $post['payment_status'] ?? '';
    if ($paymentStatus !== 'Completed') { http_response_code(200); echo 'IGNORED'; exit; }

    $receiver = strtolower(trim($post['receiver_email'] ?? $post['business'] ?? ''));
    $gross = (float)($post['mc_gross'] ?? $post['payment_gross'] ?? 0);
    $currency = $post['mc_currency'] ?? 'USD';
    $custom = (string)($post['custom'] ?? '');
    $txnId = (string)($post['txn_id'] ?? '');

    if (!verifyPaypalLicenseCustom($custom)) { http_response_code(200); echo 'OK'; exit; }
    if ($receiver !== strtolower(BACKLINK_REMOVAL_PAYPAL_EMAIL)) { Logger::error('License receiver mismatch'); http_response_code(200); echo 'OK'; exit; }
    if ($gross + 0.0001 < BACKLINK_REMOVAL_PRICE) { Logger::error('License amount too low'); http_response_code(200); echo 'OK'; exit; }

    $lic = loadData(LICENSE_FILE);
    if (!is_array($lic)) $lic = [];
    $lic['paid'] = true;
    $lic['paid_at'] = date('c');
    $lic['provider'] = 'paypal';
    $lic['provider_txn'] = $txnId;
    $lic['amount'] = $gross;
    $lic['currency'] = $currency;
    saveData(LICENSE_FILE, $lic);
    Logger::system('Backlink removal license paid via PayPal IPN');
    http_response_code(200);
    echo 'OK';
    exit;
}

function processLogoUpload(array &$settings): void {
    if (empty($_FILES['site_logo_file']) || !is_array($_FILES['site_logo_file'])) return;
    if (($_FILES['site_logo_file']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_NO_FILE) return;
    if (($_FILES['site_logo_file']['error'] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) { $_SESSION['error'] = 'Logo upload failed.'; return; }
    $tmp = (string)($_FILES['site_logo_file']['tmp_name'] ?? '');
    if ($tmp === '' || !is_uploaded_file($tmp)) { $_SESSION['error'] = 'Invalid logo upload.'; return; }

    $info = @getimagesize($tmp);
    if (!$info) { $_SESSION['error'] = 'Logo must be an image.'; return; }
    $mime = (string)($info['mime'] ?? '');

    $img = false;
    if ($mime === 'image/png' && function_exists('imagecreatefrompng')) $img = @imagecreatefrompng($tmp);
    if ($mime === 'image/jpeg' && function_exists('imagecreatefromjpeg')) $img = @imagecreatefromjpeg($tmp);
    if ($mime === 'image/webp' && function_exists('imagecreatefromwebp')) $img = @imagecreatefromwebp($tmp);
    if (!$img) { $_SESSION['error'] = 'Unsupported logo format. Use PNG/JPG/WebP.'; return; }

    $w = imagesx($img); $h = imagesy($img);
    if ($w <= 0 || $h <= 0) { imagedestroy($img); $_SESSION['error'] = 'Invalid logo image.'; return; }

    // Auto-resize (requested): keep aspect ratio, fit within a max box.
    $maxW = 320; $maxH = 80;
    $scale = min($maxW / $w, $maxH / $h, 1.0);
    $nw = max(1, (int)round($w * $scale));
    $nh = max(1, (int)round($h * $scale));

    $dst = imagecreatetruecolor($nw, $nh);
    imagealphablending($dst, false);
    imagesavealpha($dst, true);
    $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
    imagefilledrectangle($dst, 0, 0, $nw, $nh, $transparent);
    imagecopyresampled($dst, $img, 0, 0, 0, 0, $nw, $nh, $w, $h);

    if (!is_dir(LOGO_DIR)) @mkdir(LOGO_DIR, 0755, true);
    $outPath = LOGO_DIR . 'logo.png';
    if (!@imagepng($dst, $outPath, 6)) {
        $_SESSION['error'] = 'Could not save logo (check permissions).';
        imagedestroy($dst); imagedestroy($img);
        return;
    }
    imagedestroy($dst); imagedestroy($img);

    $rel = str_replace(__DIR__ . '/', '', $outPath);
    $settings['site_logo'] = $rel;
    $settings['use_logo'] = true;
    $_SESSION['message'] = 'Logo updated.';
}


// Handle image uploads
function handleImageUpload(string $itemId): array {
    $images = [];
    
    if (!isset($_FILES['images']) || !is_array($_FILES['images']['name']) || empty($_FILES['images']['name'][0])) {
        return $images;
    }
    
    for ($i = 0; $i < min(MAX_IMAGES_PER_ITEM, count($_FILES['images']['name'])); $i++) {
        if (!isset($_FILES['images']['error'][$i]) || $_FILES['images']['error'][$i] !== UPLOAD_ERR_OK) {
            continue;
        }
        
        if (!isset($_FILES['images']['type'][$i]) || !isset($_FILES['images']['size'][$i]) || !isset($_FILES['images']['tmp_name'][$i])) {
            continue;
        }
        
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!in_array($_FILES['images']['type'][$i], $allowedTypes)) continue;
        
        if ($_FILES['images']['size'][$i] > MAX_FILE_SIZE) continue;
        
        $originalName = isset($_FILES['images']['name'][$i]) ? $_FILES['images']['name'][$i] : 'image.jpg';
        $ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
        if (!in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) continue;
        
        $filename = "item_" . sanitizeInput($itemId, false) . "_" . time() . "_{$i}." . $ext;
        $destination = UPLOADS_DIR . $filename;
        
        if (@move_uploaded_file($_FILES['images']['tmp_name'][$i], $destination)) {
            $images[] = $filename;
        }
    }
    
    return $images;
}


// Helper function for formatBytes (used in mod management)
function formatBytes(int $bytes, int $precision = 2): string {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// System Status Functions
function getSystemStatus(): array {
    $status = [
        'php_version' => phpversion(),
        'memory_usage' => memory_get_usage(true),
        'memory_limit' => ini_get('memory_limit'),
        'disk_space' => disk_free_space('.'),
        'uploads_writable' => is_writable(UPLOADS_DIR),
        'data_writable' => is_writable(DATA_DIR),
        'mods_writable' => is_writable(MODS_DIR),
        'logs_writable' => is_writable(LOGS_DIR),
        'error_reporting' => error_reporting(),
        'timezone' => date_default_timezone_get(),
        'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'
    ];
    
    return $status;
}

// Initialize data files
foreach ([ITEMS_FILE, ORDERS_FILE, REVIEWS_FILE, COUPONS_FILE, USERS_FILE, TRANSACTIONS_FILE, DONATIONS_FILE, CUSTOM_MODS_FILE] as $file) {
    if (!file_exists($file)) saveData($file, []);
}

// Initialize themes
if (!file_exists(THEMES_FILE)) {
    saveData(THEMES_FILE, $defaultThemes);
} else {
    $existingThemes = loadData(THEMES_FILE);
    if (!is_array($existingThemes)) {
        $existingThemes = [];
    }
    
    $updated = false;
    foreach ($defaultThemes as $themeId => $theme) {
        if (!isset($existingThemes[$themeId]) || !is_array($existingThemes[$themeId]) || 
            !isset($existingThemes[$themeId]['colors']) || !is_array($existingThemes[$themeId]['colors'])) {
            $existingThemes[$themeId] = $theme;
            $updated = true;
        }
    }
    if ($updated) {
        saveData(THEMES_FILE, $existingThemes);
    }
}

// Initialize categories
if (!file_exists(CATEGORIES_FILE)) {
    saveData(CATEGORIES_FILE, $premiumCategories);
}

// Initialize mods
if (!file_exists(MODS_FILE)) {
    saveData(MODS_FILE, $builtinMods);
}

// Load custom mods
loadCustomMods();

// Initialize settings with proper structure
if (!file_exists(SETTINGS_FILE)) {
    $defaultSettings = [
        'site_title' => 'SmallMart',
        'admin_username' => 'admin',
        'admin_email' => 'admin@yourdomain.com',
        'use_logo' => false,
        'site_logo' => '',
        'backlink_show' => true,
        'paypal_sandbox' => false,
        'stripe_public_key' => '',
        'stripe_secret_key' => '',
        'stripe_webhook_secret' => '',
        'mail_from_email' => '',
        'mail_from_name' => 'SmallMart',
        'dev_show_email_links' => false,
        'tagline' => 'Small footprint, big features! Right click the logo and enter password: demopass to check out your new site admin panel!',
        'theme' => 'professional_dark',
        'admin_password' => password_hash(DEFAULT_ADMIN_PASSWORD, PASSWORD_DEFAULT),
        'currency' => '$',
        'timezone' => 'America/New_York',
        'items_per_page' => ITEMS_PER_PAGE,
        'max_images' => MAX_IMAGES_PER_ITEM,
        'enable_categories' => false,
        'enable_wysiwyg' => true,
        'enable_reviews' => true,
        'enable_wishlist' => true,
        'require_login_for_purchase' => true,
        'maintenance_mode' => false,
        'seo_title_suffix' => ' | SmallMart',
        'seo_description' => 'SmallMart v6 -  professional e-commerce platform with enhanced mod system',
        'seo_keywords' => 'ecommerce, marketplace, professional, paypal, advanced, premium, mods',
        'contact_email' => 'admin@yourdomain.com',
        'enabled_modes' => ['auction' => true, 'buy_now' => true, 'make_offer' => true],
        'payment_methods' => ['paypal' => true, 'stripe' => true, 'contact' => true],
        'footer_text' => '&copy; ' . COPYRIGHT_YEAR . ' SmallMart — E-Commerce Platform',
        'created' => date('Y-m-d H:i:s'),
        'mods' => $builtinMods
    ];
    saveData(SETTINGS_FILE, $defaultSettings);
}


// Initialize auth/token/license/auction/logo files
if (!is_dir(LOGO_DIR)) { @mkdir(LOGO_DIR, 0755, true); }
if (!file_exists(SECRET_FILE)) { @file_put_contents(SECRET_FILE, bin2hex(random_bytes(32))); }
foreach ([TOKENS_FILE => [], LICENSE_FILE => ['paid' => false], BIDS_FILE => [], OFFERS_FILE => [], PAGES_FILE => []] as $f => $data) {
    if (!file_exists($f)) { saveData($f, $data); }
}

// Ensure default pages exist
ensureDefaultPages();

// Initialize stats
if (!file_exists(STATS_FILE)) {
    $defaultStats = [
        'total_views' => 0,
        'unique_visitors' => 0,
        'items_sold' => 0,
        'total_revenue' => 0,
        'conversion_rate' => 0,
        'daily_stats' => [],
        'popular_items' => [],
        'search_terms' => [],
        'device_stats' => ['desktop' => 0, 'mobile' => 0, 'tablet' => 0],
        'created' => date('Y-m-d H:i:s')
    ];
    saveData(STATS_FILE, $defaultStats);
}

// Load all data
$items = loadData(ITEMS_FILE);
$settings = loadData(SETTINGS_FILE);
$orders = loadData(ORDERS_FILE);
$reviews = loadData(REVIEWS_FILE);
$categories = loadData(CATEGORIES_FILE);
$themes = loadData(THEMES_FILE);
// Auto-discover folder themes (global) (/themes/<id>/theme.json)
$themeDirs = glob(THEMES_DIR . '*', GLOB_ONLYDIR) ?: [];
foreach ($themeDirs as $td) {
    $tp = rtrim($td, "/\\") . DIRECTORY_SEPARATOR . 'theme.json';
    if (!file_exists($tp)) continue;
    $tj = json_decode(@file_get_contents($tp), true);
    if (!is_array($tj)) continue;
    $tid = $tj['id'] ?? basename($td);
    if (!isset($themes[$tid])) { $themes[$tid] = $tj; }
}

$mods = loadData(MODS_FILE);
$customMods = loadData(CUSTOM_MODS_FILE);
$stats = loadData(STATS_FILE);
$coupons = loadData(COUPONS_FILE);
$users = loadData(USERS_FILE);
ensureDefaultStoreAdminUser();
$users = loadData(USERS_FILE);
$transactions = loadData(TRANSACTIONS_FILE);
$donations = loadData(DONATIONS_FILE);

// Ensure settings has proper structure - FIXED SECTION
if (!is_array($settings) || empty($settings)) {
    $settings = [];
}

$requiredSettings = [
    'site_title' => 'SmallMart',
    'tagline' => 'Small footprint, big features!',
    'theme' => 'professional_dark',
    'currency' => '$',
    'timezone' => 'America/New_York',
    'items_per_page' => ITEMS_PER_PAGE,
    'max_images' => MAX_IMAGES_PER_ITEM,
    'enable_wysiwyg' => true,
    'enable_reviews' => true,
    'enable_wishlist' => true,
    'maintenance_mode' => false,
    'seo_title_suffix' => ' | SmallMart',
    'seo_description' => 'SmallMart v6 -  professional e-commerce platform with enhanced mod system',
    'seo_keywords' => 'ecommerce, marketplace, professional, advanced, premium, mods',
    'contact_email' => 'admin@yourdomain.com',
    'footer_text' => '&copy; ' . COPYRIGHT_YEAR . ' SmallMart -  E-Commerce Platform',
    'enabled_modes' => ['auction' => true, 'buy_now' => true, 'make_offer' => true],
    'payment_methods' => ['paypal' => true, 'contact' => true],
    'mods' => $builtinMods
];

foreach ($requiredSettings as $key => $default) {
    if (!isset($settings[$key])) {
        $settings[$key] = $default;
    }
}

if (!isset($settings['enabled_modes']) || !is_array($settings['enabled_modes'])) {
    $settings['enabled_modes'] = ['auction' => true, 'buy_now' => true, 'make_offer' => true];
}
if (!isset($settings['payment_methods']) || !is_array($settings['payment_methods'])) {
    $settings['payment_methods'] = ['paypal' => true, 'contact' => true];
}

// Ensure mods structure is properly initialized
if (!isset($settings['mods']) || !is_array($settings['mods'])) {
    $settings['mods'] = $builtinMods;
} else {
    // Merge with built-in mods to ensure all are present
    foreach ($builtinMods as $modId => $modData) {
        if (!isset($settings['mods'][$modId])) {
            $settings['mods'][$modId] = $modData;
        }
    }
}


// User Accounts (v6)
function loadUsers(): array {
    $users = loadData(USERS_FILE);
    return is_array($users) ? $users : [];
}
function saveUsers(array $users): bool {
    return saveData(USERS_FILE, $users);
}
function findUserById(string $id): ?array {
    $users = loadUsers();
    return $users[$id] ?? null;
}
function findUserIdByUsernameOrEmail(string $identifier): ?string {
    $identifier = strtolower(trim($identifier));
    foreach (loadUsers() as $id => $u) {
        $un = strtolower(trim((string)($u['username'] ?? '')));
        $em = strtolower(trim((string)($u['email'] ?? '')));
        if ($identifier !== '' && ($identifier === $un || $identifier === $em)) return (string)$id;
    }
    return null;
}
function currentUser(): ?array {
    $uid = (string)($_SESSION['user_id'] ?? '');
    if ($uid === '') return null;
    return findUserById($uid);
}
function isUserLoggedIn(): bool { return (bool)currentUser(); }
function isUserAdmin(): bool {
    $u = currentUser();
    return is_array($u) && (($u['role'] ?? '') === 'admin');
}
function requireUser(): void {
    if (!isUserLoggedIn()) {
        $_SESSION['error'] = 'Please sign in to continue.';
        header('Location: ?account=login');
        exit;
    }
}
function createUser(string $username, string $email, string $password, string $role = 'user'): array {
    $users = loadUsers();
    $id = bin2hex(random_bytes(16));
    $users[$id] = [
        'id' => $id,
        'username' => $username,
        'email' => $email,
        'pass' => password_hash($password, PASSWORD_DEFAULT),
        'role' => $role,
        'banned' => false,
        'email_verified' => false,
        'created_at' => date('c'),
        'updated_at' => date('c'),
    ];
    saveUsers($users);
    return $users[$id];
}
function ensureDefaultStoreAdminUser(): void {
    $users = loadUsers();
    foreach ($users as $u) {
        if (($u['role'] ?? '') === 'admin') return;
    }
    // Create default admin user account: admin / demopass
    $admin = createUser('admin', 'admin@localhost', DEFAULT_ADMIN_PASSWORD, 'admin');
    $users = loadUsers();
    $users[$admin['id']]['email_verified'] = true;
    saveUsers($users);
}
function setToken(string $type, string $userId, int $ttlSeconds, array $data = []): string {
    $tokens = loadData(TOKENS_FILE);
    if (!is_array($tokens)) $tokens = [];
    $token = bin2hex(random_bytes(20));
    $tokens[$token] = [
        'type' => $type,
        'user_id' => $userId,
        'data' => $data,
        'expires_at' => time() + $ttlSeconds,
        'created_at' => time(),
    ];
    saveData(TOKENS_FILE, $tokens);
    return $token;
}
function consumeToken(string $token, string $type): ?array {
    $tokens = loadData(TOKENS_FILE);
    if (!is_array($tokens)) return null;
    if (!isset($tokens[$token])) return null;
    $row = $tokens[$token];
    if (!is_array($row) || ($row['type'] ?? '') !== $type) return null;
    if ((int)($row['expires_at'] ?? 0) < time()) { unset($tokens[$token]); saveData(TOKENS_FILE, $tokens); return null; }
    unset($tokens[$token]);
    saveData(TOKENS_FILE, $tokens);
    return $row;
}
function sendAppEmail(array $settings, string $to, string $subject, string $body): bool {
    $fromEmail = trim((string)($settings['mail_from_email'] ?? ''));
    $fromName = trim((string)($settings['mail_from_name'] ?? 'SmallMart'));
    $headers = [];
    if ($fromEmail !== '') {
        $headers[] = 'From: ' . ($fromName !== '' ? $fromName : 'SmallMart') . ' <' . $fromEmail . '>';
        $headers[] = 'Reply-To: ' . $fromEmail;
    }
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'Content-Type: text/plain; charset=UTF-8';
    return @mail($to, $subject, $body, implode("\r\n", $headers));
}


// Pages (v6.1.5)
function loadPages(): array {
    $pages = loadData(PAGES_FILE);
    return is_array($pages) ? $pages : [];
}
function savePages(array $pages): bool {
    return saveData(PAGES_FILE, $pages);
}
function getPageBySlug(string $slug): ?array {
    $slug = strtolower(trim($slug));
    $pages = loadPages();
    return $pages[$slug] ?? null;
}
function upsertPage(array $page): void {
    $pages = loadPages();
    $slug = strtolower(trim((string)($page['slug'] ?? '')));
    if ($slug === '') return;
    $pages[$slug] = $page;
    savePages($pages);
}
function deletePage(string $slug): void {
    $pages = loadPages();
    $slug = strtolower(trim($slug));
    if (isset($pages[$slug])) {
        unset($pages[$slug]);
        savePages($pages);
    }
}


function ensureDefaultPages(): void {
    $pages = loadPages();
    $changed = false;
    if (!isset($pages['about'])) {
        $pages['about'] = [
            'slug' => 'about',
            'title' => 'About Us',
            'show_in_nav' => false,
            'show_in_footer' => true,
            'content' => sanitizeInput("<p><strong>SmallMart</strong> is a lightweight, self-hosted store script built for fast installs and easy customization.</p>\n<ul>\n  <li>Multi-mode selling: Buy Now, Auctions, and Offers</li>\n  <li>Theme system + Theme Builder</li>\n  <li>Mod system with hooks and package mods</li>\n  <li>Admin tools for items, orders, settings, backups, logs</li>\n</ul>\n<p>This store is operated by an independent site owner. Product availability, fulfillment, and policies are determined by the store owner/seller.</p>", true),
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];
        $changed = true;
    }
    if (!isset($pages['tos'])) {
        $pages['tos'] = [
            'slug' => 'tos',
            'title' => 'Terms of Service',
            'show_in_nav' => false,
            'show_in_footer' => true,
            'content' => sanitizeInput("<p>These Terms govern your use of this website and any purchases made through it. If you do not agree, do not use the site.</p>\n<h3>1) Eligibility & Accounts</h3>\n<ul>\n  <li>You are responsible for maintaining the confidentiality of your account credentials.</li>\n  <li>You agree to provide accurate information and keep it up to date.</li>\n</ul>\n<h3>2) Orders, Payments, and Digital Goods</h3>\n<ul>\n  <li>All prices are shown in the store currency at time of purchase.</li>\n  <li>Payment processing may be handled by third-party providers (e.g., PayPal/Stripe).</li>\n  <li>For digital products, delivery may occur instantly after payment confirmation.</li>\n</ul>\n<h3>3) Refunds & Chargebacks</h3>\n<ul>\n  <li>Refund eligibility depends on the seller\u2019s policy shown at checkout or in the listing.</li>\n  <li>Abusive chargebacks may result in account restriction.</li>\n</ul>\n<h3>4) Prohibited Use</h3>\n<ul>\n  <li>No fraud, hacking, scraping, or abuse of site features.</li>\n  <li>No uploading content you do not own or have rights to use.</li>\n</ul>\n<h3>5) Content & Intellectual Property</h3>\n<ul>\n  <li>You retain ownership of your content, but grant the site a license to display it as needed to operate the service.</li>\n</ul>\n<h3>6) Disclaimers & Limitation of Liability</h3>\n<ul>\n  <li>The site is provided \u201cas is\u201d without warranties to the fullest extent permitted by law.</li>\n  <li>To the fullest extent permitted by law, the site owner is not liable for indirect or consequential damages.</li>\n</ul>\n<h3>7) Changes</h3>\n<p>We may update these Terms. Continued use after changes means you accept the updated Terms.</p>\n<p><strong>Contact:</strong> Use the seller contact method available on product pages/checkout.</p>", true),
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];
        $changed = true;
    }
    if ($changed) savePages($pages);
}

// Action processing
$action = safeGet($_GET, 'action') ?: safeGet($_POST, 'action');
$method = $_SERVER['REQUEST_METHOD'];

// Log system initialization
// Webhook endpoints (must exit quickly)
if (isset($_GET['paypal_ipn'])) {
    // PayPal IPN for backlink license
    handlePaypalLicenseIpn($settings);
    exit;
}
Logger::system('System initialized - Version: ' . VERSION);

if ($method === 'POST') {
    switch ($action) {
        case 'login':
            $password = safeGet($_POST, 'password');
            if (password_verify($password, $settings['admin_password'])) {
                $_SESSION['admin'] = true;
                $_SESSION['message'] = 'Welcome to SmallMart v6!';
                Logger::admin('Admin login successful');
                header('Location: ?admin=dashboard');
                exit;
            } else {
                $_SESSION['error'] = 'Invalid password';
                Logger::admin('Admin login failed - invalid password');
                header('Location: ?admin_login=1');
                exit;
            }
            break;

        case 'user_register':
            // Public registration
            $username = trim((string)safeGet($_POST, 'username'));
            $email = trim((string)safeGet($_POST, 'email'));
            $password = (string)safeGet($_POST, 'password');
            if ($username === '' || $email === '' || $password === '') {
                $_SESSION['error'] = 'Please fill all fields.';
                header('Location: ?account=register');
                exit;
            }
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['error'] = 'Invalid email address.';
                header('Location: ?account=register');
                exit;
            }
            if (strlen($username) < 3 || strlen($username) > 32 || !preg_match('/^[a-zA-Z0-9_\-]+$/', $username)) {
                $_SESSION['error'] = 'Username must be 3-32 characters and use letters/numbers/_/- only.';
                header('Location: ?account=register');
                exit;
            }
            $users = loadUsers();
            foreach ($users as $u) {
                if (strtolower($u['username'] ?? '') === strtolower($username)) {
                    $_SESSION['error'] = 'Username already taken.';
                    header('Location: ?account=register');
                    exit;
                }
                if (strtolower($u['email'] ?? '') === strtolower($email)) {
                    $_SESSION['error'] = 'Email already in use.';
                    header('Location: ?account=register');
                    exit;
                }
            }
            $user = createUser($username, $email, $password, 'user');
            $_SESSION['user_id'] = $user['id'];

            // Email verification
            $token = setToken('email_verify', $user['id'], EMAIL_VERIFY_TTL_SECONDS);
            $verifyLink = baseUrl() . '?account=verify_email&token=' . urlencode($token);
            $sent = false;
            if (!empty($settings['mail_from_email'])) {
                $sent = sendAppEmail($settings, $email, 'Verify your email', "Welcome to {$settings['site_title']}!\n\nVerify your email: $verifyLink\n");
            }
            if (!$sent && !empty($settings['dev_show_email_links'])) {
                $_SESSION['message'] = 'Account created! Dev link: ' . $verifyLink;
            } else {
                $_SESSION['message'] = 'Account created! Please check your email to verify.';
            }
            header('Location: ?account=settings');
            exit;

        case 'user_login':
            $identifier = trim((string)safeGet($_POST, 'identifier'));
            $password = (string)safeGet($_POST, 'password');
            $uid = findUserIdByUsernameOrEmail($identifier);
            if (!$uid) {
                $_SESSION['error'] = 'Invalid login.';
                header('Location: ?account=login');
                exit;
            }
            $u = findUserById($uid);
            if (!is_array($u) || !empty($u['banned'])) {
                $_SESSION['error'] = 'Account disabled.';
                header('Location: ?account=login');
                exit;
            }
            if (!password_verify($password, (string)($u['pass'] ?? ''))) {
                $_SESSION['error'] = 'Invalid login.';
                header('Location: ?account=login');
                exit;
            }
            $_SESSION['user_id'] = $uid;
            $_SESSION['message'] = 'Signed in successfully.';
            header('Location: ?');
            exit;

        case 'user_logout':
            
            
            // Also clear any admin session flags
            unset($_SESSION['admin_logged_in'], $_SESSION['admin_user'], $_SESSION['is_admin'], $_SESSION['admin_id']);
// Log out both user and admin (if active)
            unset($_SESSION['admin_logged_in'], $_SESSION['admin_user'], $_SESSION['is_admin'], $_SESSION['admin_id']);
unset($_SESSION['user_id']);
            $_SESSION['message'] = 'Signed out.';
            header('Location: ?');
            exit;
        case 'admin_logout':
            // Admin logout (requires valid CSRF)
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?');
                exit;
            }
            $_SESSION = [];
            if (ini_get('session.use_cookies')) {
                $params = session_get_cookie_params();
                setcookie(session_name(), '', time() - 42000, $params['path'], $params['domain'], $params['secure'], $params['httponly']);
            }
            session_destroy();
            session_start();
            header('Location: ?');
            exit;



        case 'user_update_account':
            requireUser();
            $u = currentUser();
            $uid = (string)$u['id'];
            $newUsername = trim((string)safeGet($_POST, 'username', $u['username'] ?? ''));
            $newEmail = trim((string)safeGet($_POST, 'email', $u['email'] ?? ''));
            $newPass = (string)safeGet($_POST, 'new_password');
            $users = loadUsers();

            // Unique checks
            foreach ($users as $id => $row) {
                if ($id === $uid) continue;
                if (strtolower($row['username'] ?? '') === strtolower($newUsername)) {
                    $_SESSION['error'] = 'Username already taken.';
                    header('Location: ?account=settings');
                    exit;
                }
                if (strtolower($row['email'] ?? '') === strtolower($newEmail)) {
                    $_SESSION['error'] = 'Email already in use.';
                    header('Location: ?account=settings');
                    exit;
                }
            }
            if ($newUsername === '' || strlen($newUsername) < 3 || strlen($newUsername) > 32 || !preg_match('/^[a-zA-Z0-9_\-]+$/', $newUsername)) {
                $_SESSION['error'] = 'Invalid username.';
                header('Location: ?account=settings');
                exit;
            }
            if (!filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['error'] = 'Invalid email.';
                header('Location: ?account=settings');
                exit;
            }
            $users[$uid]['username'] = $newUsername;
            if (strtolower($users[$uid]['email'] ?? '') !== strtolower($newEmail)) {
                $users[$uid]['email'] = $newEmail;
                $users[$uid]['email_verified'] = false;
                $token = setToken('email_verify', $uid, EMAIL_VERIFY_TTL_SECONDS);
                $verifyLink = baseUrl() . '?account=verify_email&token=' . urlencode($token);
                if (!empty($settings['mail_from_email'])) {
                    sendAppEmail($settings, $newEmail, 'Verify your email', "Verify your email: $verifyLink\n");
                    $_SESSION['message'] = 'Profile updated. Please verify your new email.';
                } else if (!empty($settings['dev_show_email_links'])) {
                    $_SESSION['message'] = 'Profile updated. Dev link: ' . $verifyLink;
                } else {
                    $_SESSION['message'] = 'Profile updated.';
                }
            } else {
                $_SESSION['message'] = 'Profile updated.';
            }
            if ($newPass !== '') {
                $users[$uid]['pass'] = password_hash($newPass, PASSWORD_DEFAULT);
                $_SESSION['message'] = 'Profile updated (password changed).';
            }
            $users[$uid]['updated_at'] = date('c');
            saveUsers($users);
            header('Location: ?account=settings');
            exit;

        case 'user_request_reset':
            $email = trim((string)safeGet($_POST, 'email'));
            $uid = findUserIdByUsernameOrEmail($email);
            // Always respond success (no enumeration)
            if ($uid) {
                $tok = setToken('pw_reset', $uid, PASSWORD_RESET_TTL_SECONDS);
                $link = baseUrl() . '?account=reset_password&token=' . urlencode($tok);
                if (!empty($settings['mail_from_email'])) {
                    sendAppEmail($settings, $email, 'Password reset', "Reset your password: $link\nThis link expires in 1 hour.\n");
                } else if (!empty($settings['dev_show_email_links'])) {
                    $_SESSION['message'] = 'Dev reset link: ' . $link;
                }
            }
            if (empty($_SESSION['message'])) $_SESSION['message'] = 'If that email exists, a reset link has been sent.';
            header('Location: ?account=login');
            exit;

        case 'user_reset_password':
            $token = trim((string)safeGet($_POST, 'token'));
            $newPass = (string)safeGet($_POST, 'new_password');
            $row = consumeToken($token, 'pw_reset');
            if (!$row || $newPass === '') {
                $_SESSION['error'] = 'Invalid or expired reset token.';
                header('Location: ?account=login');
                exit;
            }
            $uid = (string)($row['user_id'] ?? '');
            $users = loadUsers();
            if (!isset($users[$uid])) {
                $_SESSION['error'] = 'Invalid reset token.';
                header('Location: ?account=login');
                exit;
            }
            $users[$uid]['pass'] = password_hash($newPass, PASSWORD_DEFAULT);
            $users[$uid]['updated_at'] = date('c');
            saveUsers($users);
            $_SESSION['message'] = 'Password reset successfully. Please sign in.';
            header('Location: ?account=login');
            exit;

        case 'admin_user_update':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=users');
                exit;
            }

        case 'admin_save_page':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=pages');
                exit;
            }
            $slug = strtolower(trim((string)safeGet($_POST, 'slug')));
            $title = trim((string)safeGet($_POST, 'title'));
            $showInNav = isset($_POST['show_in_nav']);
            $showInFooter = isset($_POST['show_in_footer']);
            $content = (string)safeGet($_POST, 'content');
            if ($slug === '' || !preg_match('/^[a-z0-9\-]+$/', $slug)) {
                $_SESSION['error'] = 'Slug must use lowercase letters, numbers, and hyphens only.';
                header('Location: ?admin=pages');
                exit;
            }
            if ($title === '') {
                $_SESSION['error'] = 'Title is required.';
                header('Location: ?admin=pages&edit=' . urlencode($slug));
                exit;
            }
            $page = [
                'slug' => $slug,
                'title' => sanitizeInput($title, false),
                'show_in_nav' => $showInNav,
                'show_in_footer' => $showInFooter,
                'content' => sanitizeInput($content, true),
                'updated_at' => date('c'),
            ];
            $existing = getPageBySlug($slug);
            if (!$existing) $page['created_at'] = date('c'); else $page['created_at'] = $existing['created_at'] ?? date('c');
            upsertPage($page);
            $_SESSION['message'] = 'Page saved.';
            header('Location: ?admin=pages');
            exit;

        case 'admin_delete_page':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=pages');
                exit;
            }
            $slug = strtolower(trim((string)safeGet($_POST, 'slug')));
            deletePage($slug);
            $_SESSION['message'] = 'Page deleted.';
            header('Location: ?admin=pages');
            exit;
            $uid = (string)safeGet($_POST, 'user_id');
            $op = (string)safeGet($_POST, 'op');
            $users = loadUsers();
            if (!isset($users[$uid])) { $_SESSION['error'] = 'User not found.'; header('Location: ?admin=users'); exit; }

            if ($op === 'toggle_ban') {
                $users[$uid]['banned'] = !empty($users[$uid]['banned']) ? false : true;
                $_SESSION['message'] = $users[$uid]['banned'] ? 'User banned.' : 'User unbanned.';
            } elseif ($op === 'toggle_admin') {
                $users[$uid]['role'] = (($users[$uid]['role'] ?? 'user') === 'admin') ? 'user' : 'admin';
                $_SESSION['message'] = 'User role updated.';
            } elseif ($op === 'reset_pass') {
                $temp = 'SM' . substr(bin2hex(random_bytes(8)), 0, 10);
                $users[$uid]['pass'] = password_hash($temp, PASSWORD_DEFAULT);
                $_SESSION['message'] = 'Temporary password: ' . $temp;
            }
            $users[$uid]['updated_at'] = date('c');
            saveUsers($users);
            header('Location: ?admin=users');
            exit;


            
        case 'save_item':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=items');
                exit;
            }
            
            $id = safeGet($_POST, 'id') ?: generateId();
            $images = handleImageUpload($id);
            
            if (!empty($_POST['existing_images'])) {
                $existing = is_array($_POST['existing_images']) ? $_POST['existing_images'] : [];
                $images = array_merge($existing, $images);
            }
            
            $item = [
                'id' => $id,
                'title' => sanitizeInput(safeGet($_POST, 'title'), false),
                'description' => sanitizeInput(safeGet($_POST, 'description'), true),
                'price' => sanitizeInput(safeGet($_POST, 'price'), false),
                'type' => sanitizeInput(safeGet($_POST, 'type', 'buy_now'), false),
                'category' => sanitizeInput(safeGet($_POST, 'category'), false),
                'condition' => sanitizeInput(safeGet($_POST, 'condition', 'new'), false),
                'stock' => intval(safeGet($_POST, 'stock', 1)),
                'location' => sanitizeInput(safeGet($_POST, 'location'), false),
                'shipping_cost' => floatval(safeGet($_POST, 'shipping_cost', 0)),
                'tags' => sanitizeInput(safeGet($_POST, 'tags'), false),
                'featured' => isset($_POST['featured']),
                'enable_paypal' => isset($_POST['enable_paypal']),
                'enable_contact' => isset($_POST['enable_contact']),
                'images' => $images,
                'created' => safeGet($_POST, 'created') ?: time(),
                'updated' => time(),
                'views' => intval(safeGet($_POST, 'views', 0)),
                'status' => 'active'
            ];
            
            $found = false;
            foreach ($items as $key => $existingItem) {
                if ($existingItem['id'] === $id) {
                    $items[$key] = $item;
                    $found = true;
                    break;
}
            }
            
            if (!$found) {
                $items[] = $item;
            }
            
            saveData(ITEMS_FILE, $items);
            Logger::admin('Item saved: ' . $item['title'] . ' (ID: ' . $id . ')');
            
            $_SESSION['message'] = 'Item saved successfully!';
            header('Location: ?admin=items');
            exit;
            break;
            
        case 'save_theme':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=theme_builder');
                exit;
            }
            
            $themeId = sanitizeInput(safeGet($_POST, 'theme_id')) ?: 'custom_' . time();
            $themeName = sanitizeInput(safeGet($_POST, 'theme_name')) ?: 'Custom Theme';
            
            $customTheme = [
                'name' => $themeName,
                'category' => 'custom',
                'is_custom' => true,
                'created' => date('Y-m-d H:i:s'),
                'colors' => [
                    'primary' => sanitizeInput(safeGet($_POST, 'primary', '#ffcc00')),
                    'secondary' => sanitizeInput(safeGet($_POST, 'secondary', '#ff9900')),
                    'accent' => sanitizeInput(safeGet($_POST, 'accent', '#ffd700')),
                    'background' => sanitizeInput(safeGet($_POST, 'background', '#1a1a1a')),
                    'surface' => sanitizeInput(safeGet($_POST, 'surface', 'rgba(45, 45, 45, 0.95)')),
                    'card' => sanitizeInput(safeGet($_POST, 'card', 'rgba(60, 60, 60, 0.9)')),
                    'text' => sanitizeInput(safeGet($_POST, 'text', '#ffffff')),
                    'text_secondary' => sanitizeInput(safeGet($_POST, 'text_secondary', '#e0e0e0')),
                    'border' => sanitizeInput(safeGet($_POST, 'border', 'rgba(255, 204, 0, 0.2)'))
                ],
                'fonts' => [
                    'primary' => sanitizeInput(safeGet($_POST, 'font_primary', "'Inter', sans-serif")),
                    'heading' => sanitizeInput(safeGet($_POST, 'font_heading', "'Poppins', sans-serif"))
                ]
            ];
            
            $themes[$themeId] = $customTheme;
            saveData(THEMES_FILE, $themes);
            Logger::admin('Custom theme created: ' . $themeName);
            
            $_SESSION['message'] = 'Custom theme saved successfully!';
            header('Location: ?admin=theme_builder');
            exit;
            break;
            
        case 'upload_mod':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=mods');
                exit;
            }
            
            $result = handleModUpload();
            if ($result['success']) {
                $_SESSION['message'] = $result['message'];
                Logger::admin('Mod uploaded: ' . $result['mod_id']);
            } else {
                $_SESSION['error'] = $result['message'];
                Logger::error('Mod upload failed: ' . $result['message']);
            }
            
            header('Location: ?admin=mods');
            exit;
            break;
            
        case 'import_theme':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=theme_builder');
                exit;
            }
            
            $themeData = safeGet($_POST, 'theme_data');
            if (empty($themeData)) {
                $_SESSION['error'] = 'No theme data provided';
                header('Location: ?admin=theme_builder');
                exit;
            }
            
            $result = importTheme($themeData);
            if ($result['success']) {
                $_SESSION['message'] = $result['message'];
                Logger::admin('Theme imported: ' . $result['theme_id']);
            } else {
                $_SESSION['error'] = $result['message'];
                Logger::error('Theme import failed: ' . $result['message']);
            }
            
            header('Location: ?admin=theme_builder');
            exit;
            break;
            
        case 'remove_backlink':
            requireAdmin();
            $_SESSION['error'] = 'Backlink removal now requires a $5 license. Go to Admin → License.';
            header('Location: ?admin=license');
            exit;
            break;

case 'save_settings':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=settings');
                exit;
            }
            
            // Handle theme-only change
            if (isset($_POST['change_theme_only']) && isset($_POST['theme'])) {
                $newTheme = sanitizeInput($_POST['theme'], false);
                if (isset($themes[$newTheme])) {
                    $settings['theme'] = $newTheme;
                    if (saveData(SETTINGS_FILE, $settings)) {
                        $_SESSION['message'] = 'Theme changed successfully!';
                        Logger::admin('Theme changed to: ' . $newTheme);
                    } else {
                        $_SESSION['error'] = 'Failed to save theme change';
                    }
                    header('Location: ?admin=theme_builder');
                    exit;
                } else {
                    $_SESSION['error'] = 'Invalid theme selected';
                    header('Location: ?admin=theme_builder');
                    exit;
                }
            }
            
            // Load current settings to preserve structure
            $currentSettings = loadData(SETTINGS_FILE);
            if (!is_array($currentSettings)) {
                $currentSettings = [];
            }
            
            // Password change
            if (!empty($_POST['new_password'])) {
                $currentSettings['admin_password'] = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
                Logger::admin('Admin password changed');
            }
            
            // Basic settings - update directly in currentSettings to preserve structure
            $currentSettings['site_title'] = sanitizeInput(safeGet($_POST, 'site_title'), false);
            $currentSettings['tagline'] = sanitizeInput(safeGet($_POST, 'tagline'), false);
            $currentSettings['theme'] = sanitizeInput(safeGet($_POST, 'theme', $currentSettings['theme'] ?? 'professional_dark'), false);
            $currentSettings['currency'] = sanitizeInput(safeGet($_POST, 'currency', '$'), false);
            $currentSettings['timezone'] = sanitizeInput(safeGet($_POST, 'timezone', 'America/New_York'), false);
            $currentSettings['items_per_page'] = intval(safeGet($_POST, 'items_per_page', ITEMS_PER_PAGE));
            $currentSettings['max_images'] = intval(safeGet($_POST, 'max_images', MAX_IMAGES_PER_ITEM));
            $currentSettings['enable_wysiwyg'] = isset($_POST['enable_wysiwyg']);
            $currentSettings['enable_reviews'] = isset($_POST['enable_reviews']);
            $currentSettings['enable_wishlist'] = isset($_POST['enable_wishlist']);
            $currentSettings['enable_categories'] = isset($_POST['enable_categories']);
            $currentSettings['require_login_for_purchase'] = isset($_POST['require_login_for_purchase']);
            $currentSettings['maintenance_mode'] = isset($_POST['maintenance_mode']);
            $currentSettings['seo_title_suffix'] = sanitizeInput(safeGet($_POST, 'seo_title_suffix'), false);
            $currentSettings['seo_description'] = sanitizeInput(safeGet($_POST, 'seo_description'), false);
            $currentSettings['seo_keywords'] = sanitizeInput(safeGet($_POST, 'seo_keywords'), false);
            $currentSettings['contact_email'] = sanitizeInput(safeGet($_POST, 'contact_email'), false);
            $currentSettings['footer_text'] = sanitizeInput(safeGet($_POST, 'footer_text'), true);
            $currentSettings['powered_by'] = sanitizeInput(safeGet($_POST, 'powered_by'), true);
            // Admin profile fields
            $currentSettings['admin_username'] = sanitizeInput(safeGet($_POST, 'admin_username', $currentSettings['admin_username'] ?? 'admin'), false);
            $currentSettings['admin_email'] = sanitizeInput(safeGet($_POST, 'admin_email', $currentSettings['admin_email'] ?? ''), false);

            // Backlink toggle (only if license paid)
            if (isDonationComplete()) {
                $currentSettings['backlink_show'] = isset($_POST['backlink_show']);
            } else {
                $currentSettings['backlink_show'] = true;
            }

            // Logo toggle
            $currentSettings['use_logo'] = isset($_POST['use_logo']);

            // Stripe keys
            $currentSettings['stripe_public_key'] = sanitizeInput(safeGet($_POST, 'stripe_public_key', $currentSettings['stripe_public_key'] ?? ''), false);
            $currentSettings['stripe_secret_key'] = sanitizeInput(safeGet($_POST, 'stripe_secret_key', $currentSettings['stripe_secret_key'] ?? ''), false);
            $currentSettings['stripe_webhook_secret'] = sanitizeInput(safeGet($_POST, 'stripe_webhook_secret', $currentSettings['stripe_webhook_secret'] ?? ''), false);

            // Mail
            $currentSettings['mail_from_email'] = sanitizeInput(safeGet($_POST, 'mail_from_email', $currentSettings['mail_from_email'] ?? ''), false);
            $currentSettings['mail_from_name'] = sanitizeInput(safeGet($_POST, 'mail_from_name', $currentSettings['mail_from_name'] ?? 'SmallMart'), false);
            $currentSettings['dev_show_email_links'] = isset($_POST['dev_show_email_links']);

            
            // Enabled modes
            $currentSettings['enabled_modes'] = [
                'auction' => isset($_POST['mode_auction']),
                'buy_now' => isset($_POST['mode_buy_now']),
                'make_offer' => isset($_POST['mode_make_offer'])
            ];
            
            // Payment methods
            $currentSettings['payment_methods'] = [
                'paypal' => isset($_POST['payment_paypal']),
                'stripe' => isset($_POST['payment_stripe']),
                'contact' => isset($_POST['payment_contact'])
            ];
            
            // PayPal settings - ensure proper structure
            if (!isset($currentSettings['mods'])) {
                $currentSettings['mods'] = $builtinMods;
            }
            if (!isset($currentSettings['mods']['paypal_integration'])) {
                $currentSettings['mods']['paypal_integration'] = $builtinMods['paypal_integration'];
            }
            if (!isset($currentSettings['mods']['paypal_integration']['settings'])) {
                $currentSettings['mods']['paypal_integration']['settings'] = [];
            }
            
            if (isset($_POST['paypal_email'])) {
                $currentSettings['mods']['paypal_integration']['settings']['paypal_email'] = sanitizeInput($_POST['paypal_email']);
            }
            
            $currentSettings['mods']['paypal_integration']['settings']['sandbox_mode'] = isset($_POST['paypal_sandbox_mode']);
            
            processLogoUpload($currentSettings);
            
            // Save settings
            if (saveData(SETTINGS_FILE, $currentSettings)) {
                // Reload settings to ensure they are applied
                $settings = $currentSettings;
                $_SESSION['message'] = 'Settings updated and applied successfully!';
                Logger::admin('Settings updated');
            } else {
                $_SESSION['error'] = 'Failed to save settings. Please check file permissions.';
                Logger::error('Failed to save settings');
            }
            
            header('Location: ?admin=settings');
            exit;
            break;
            
        case 'save_mod_settings':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=mods');
                exit;
            }
            
            $modId = safeGet($_POST, 'mod_id');
            if ($modId && isset($mods[$modId])) {
                // Update mod settings
                foreach ($_POST as $key => $value) {
                    if (strpos($key, 'mod_setting_') === 0) {
                        $settingKey = substr($key, 12); // Remove 'mod_setting_' prefix
                        $mods[$modId]['settings'][$settingKey] = $value;
                    }
                }
                
                // Handle checkboxes
                foreach ($mods[$modId]['settings'] as $key => $currentValue) {
                    if (is_bool($currentValue) && !isset($_POST['mod_setting_' . $key])) {
                        $mods[$modId]['settings'][$key] = false;
                    }
                }
                
                saveData(MODS_FILE, $mods);
                Logger::admin('Mod settings updated: ' . $modId);
                $_SESSION['message'] = 'Mod settings saved successfully!';
            }
            header('Location: ?admin=mods&edit=' . $modId);
            exit;
            break;
            
        case 'clear_log':
            requireAdmin();
            if (!validateCSRF(safeGet($_POST, 'csrf_token'))) {
                $_SESSION['error'] = 'Invalid security token';
                header('Location: ?admin=console');
                exit;
            }
            
            $logType = safeGet($_POST, 'log_type');
            $logFiles = [
                'system' => SYSTEM_LOG_FILE,
                'error' => ERROR_LOG_FILE,
                'admin' => ADMIN_LOG_FILE,
                'mod' => MOD_LOG_FILE
            ];
            
            if (isset($logFiles[$logType])) {
                if (Logger::clearLog($logFiles[$logType])) {
                    Logger::admin("Cleared $logType log");
                    $_SESSION['message'] = ucfirst($logType) . ' log cleared successfully!';
                } else {
                    $_SESSION['error'] = 'Failed to clear log file';
                }
            }
            
            header('Location: ?admin=console');
            exit;
            break;
    }
}

// Handle GET actions
if ($method === 'GET') {
    if (isset($_GET['paypal_success'])) {
        $itemId = safeGet($_GET, 'item_id');
        if ($itemId) {
            $orderRecord = [
                'id' => generateId('order'),
                'item_id' => $itemId,
                'status' => 'completed',
                'payment_method' => 'paypal',
                'created' => date('Y-m-d H:i:s'),
                'completed_at' => date('Y-m-d H:i:s')
            ];
            $orders[] = $orderRecord;
            saveData(ORDERS_FILE, $orders);
            
            if (!is_array($stats)) $stats = [];
            if (!isset($stats['items_sold'])) $stats['items_sold'] = 0;
            $stats['items_sold']++;
            saveData(STATS_FILE, $stats);
            
            Logger::system('PayPal payment completed for item: ' . $itemId);
            $_SESSION['message'] = 'Payment completed successfully! Thank you for your purchase.';
        }
        header('Location: ?view=' . $itemId);
        exit;
    }
    
    if (isset($_GET['donation_completed'])) {
        if (recordDonation(false)) {
            $_SESSION['message'] = 'Thank you for your donation! The backlink has been removed automatically.';
        } else {
            $_SESSION['error'] = 'Failed to process donation confirmation. Please contact support.';
        }
        header('Location: ?admin=settings');
        exit;
    }
    
    if ($action) {
        switch ($action) {
            case 'backup':
                requireAdmin();
                if (createBackup()) {
                    $_SESSION['message'] = 'Backup created successfully!';
                } else {
                    $_SESSION['error'] = 'Failed to create backup';
                }
                header('Location: ?admin=dashboard');
                exit;
                break;
                
            case 'delete_item':
                requireAdmin();
                $itemId = safeGet($_GET, 'id');
                if ($itemId) {
                    foreach ($items as $key => $item) {
                        if ($item['id'] === $itemId) {
                            // Delete associated images
                            if (!empty($item['images'])) {
                                foreach ($item['images'] as $image) {
                                    $imagePath = UPLOADS_DIR . $image;
                                    if (file_exists($imagePath)) {
                                        @unlink($imagePath);
                                    }
                                }
                            }
                            Logger::admin('Item deleted: ' . $item['title'] . ' (ID: ' . $itemId . ')');
                            unset($items[$key]);
                            break;
                        }
                    }
                    saveData(ITEMS_FILE, array_values($items));
                    $_SESSION['message'] = 'Item deleted successfully!';
                }
                header('Location: ?admin=items');
                exit;
                break;
                
            case 'toggle_mod':
                requireAdmin();
                $modId = safeGet($_GET, 'id');
                if ($modId && isset($mods[$modId])) {
                    $mods[$modId]['enabled'] = !($mods[$modId]['enabled'] ?? false);
                    $mods[$modId]['status'] = $mods[$modId]['enabled'] ? 'active' : 'inactive';
                    saveData(MODS_FILE, $mods);
                    $status = $mods[$modId]['enabled'] ? 'enabled' : 'disabled';
                    Logger::admin("Mod $status: $modId");
                    $_SESSION['message'] = 'Mod ' . $status . ' successfully!';
                }
                header('Location: ?admin=mods');
                exit;
                break;
                
            case 'clear_orders':
                requireAdmin();
                saveData(ORDERS_FILE, []);
                saveData(TRANSACTIONS_FILE, []);
                Logger::admin('Order history cleared');
                $_SESSION['message'] = 'Order history cleared successfully!';
                header('Location: ?admin=orders');
                exit;
                break;
                
            case 'delete_theme':
                requireAdmin();
                $themeId = safeGet($_GET, 'id');
                if ($themeId && isset($themes[$themeId]) && ($themes[$themeId]['is_custom'] ?? false)) {
                    unset($themes[$themeId]);
                    saveData(THEMES_FILE, $themes);
                    Logger::admin('Custom theme deleted: ' . $themeId);
                    $_SESSION['message'] = 'Custom theme deleted successfully!';
                }
                header('Location: ?admin=theme_builder');
                exit;
                break;
                
            case 'export_theme':
                requireAdmin();
                $themeId = safeGet($_GET, 'id');
                if ($themeId && isset($themes[$themeId])) {
                    $exportData = exportTheme($themeId);
                    if ($exportData) {
                        Logger::admin('Theme exported: ' . $themeId);
                        header('Content-Type: application/octet-stream');
                        header('Content-Disposition: attachment; filename="' . $themeId . '_theme.txt"');
                        echo $exportData;
                        exit;
                    }
                }
                $_SESSION['error'] = 'Failed to export theme';
                header('Location: ?admin=theme_builder');
                exit;
                break;
                
          case 'delete_mod':
    requireAdmin();
    $modId = safeGet($_GET, 'id');
    if ($modId) {
        // Delete the physical file
        $modFile = MODS_DIR . $modId . '.php';
        if (file_exists($modFile)) {
            @unlink($modFile);
        }
        
        // Remove from custom mods registry
        $customMods = loadData(CUSTOM_MODS_FILE);
        if (isset($customMods[$modId])) {
            unset($customMods[$modId]);
            saveData(CUSTOM_MODS_FILE, $customMods);
        }
        
        // Remove from main mods registry
        $mods = loadData(MODS_FILE);
        if (isset($mods[$modId])) {
            unset($mods[$modId]);
            saveData(MODS_FILE, $mods);
        }
        
        Logger::admin('Custom mod deleted: ' . $modId);
        $_SESSION['message'] = 'Custom mod deleted successfully!';
    }
    header('Location: ?admin=mods');
    exit;
    break;
        }
    }
}

// Set timezone
$timezone = isset($settings['timezone']) && !empty($settings['timezone']) ? $settings['timezone'] : 'America/New_York';
if (@date_default_timezone_set($timezone) === false) {
    @date_default_timezone_set('America/New_York');
}

// Track analytics
if (!isset($_SESSION['view_tracked'])) {
    if (!is_array($stats)) $stats = [];
    if (!isset($stats['total_views'])) $stats['total_views'] = 0;
    if (!isset($stats['daily_stats']) || !is_array($stats['daily_stats'])) $stats['daily_stats'] = [];
    if (!isset($stats['device_stats']) || !is_array($stats['device_stats'])) {
        $stats['device_stats'] = ['desktop' => 0, 'mobile' => 0, 'tablet' => 0];
    }
    
    $stats['total_views']++;
    $today = date('Y-m-d');
    if (!isset($stats['daily_stats'][$today]) || !is_array($stats['daily_stats'][$today])) {
        $stats['daily_stats'][$today] = ['views' => 0, 'visitors' => 0];
    }
    $stats['daily_stats'][$today]['views']++;
    
    $userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    if (preg_match('/mobile/i', $userAgent)) {
        $stats['device_stats']['mobile']++;
    } elseif (preg_match('/tablet/i', $userAgent)) {
        $stats['device_stats']['tablet']++;
    } else {
        $stats['device_stats']['desktop']++;
    }
    
    saveData(STATS_FILE, $stats);
    $_SESSION['view_tracked'] = true;
}

// Check maintenance mode
if (($settings['maintenance_mode'] ?? false) && !isAdmin()) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Maintenance - <?php echo sanitizeInput($settings['site_title'], false); ?></title>
        <style>
            body { font-family: -apple-system, BlinkMacSystemFont, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-align: center; padding: 50px; min-height: 100vh; display: flex; align-items: center; justify-content: center; margin: 0; }
            .maintenance { background: rgba(255,255,255,0.1); backdrop-filter: blur(20px); padding: 60px 40px; border-radius: 20px; max-width: 500px; }
            h1 { font-size: 3rem; margin-bottom: 20px; }
            p { font-size: 1.2rem; opacity: 0.9; line-height: 1.6; }
        </style>
    </head>
    <body>
        <div class="maintenance">
            <h1>🔧</h1>
            <h1>Maintenance Mode</h1>
            <p>We're currently performing scheduled maintenance to improve your experience. Please check back soon!</p>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Handle admin login
if (isset($_GET['admin_login'])) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Login - <?php echo sanitizeInput($settings['site_title'], false); ?></title>
        <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
        <style>
            body { 
                font-family: 'Inter', sans-serif; 
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: #fff; 
                padding: 20px; 
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0;
            }
            .login-container { 
                max-width: 450px; 
                width: 100%;
                background: rgba(255, 255, 255, 0.1);
                backdrop-filter: blur(30px);
                padding: 50px 40px; 
                border-radius: 24px;
                box-shadow: 0 25px 50px rgba(0,0,0,0.3);
                border: 1px solid rgba(255,255,255,0.2);
            }
            .logo {
                text-align: center;
                font-size: 2.5rem;
                font-weight: 700;
                margin-bottom: 10px;
                background: linear-gradient(135deg, #ffcc00, #ff9900);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
            }
            .version {
                text-align: center;
                font-size: 0.9rem;
                opacity: 0.8;
                margin-bottom: 40px;
            }
            .form-group { margin-bottom: 25px; }
            .form-label { 
                display: block; 
                margin-bottom: 8px; 
                font-weight: 500;
                font-size: 0.9rem;
            }
            .form-control { 
                width: 100%; 
                padding: 18px 20px; 
                background: rgba(255, 255, 255, 0.1);
                border: 2px solid rgba(255,255,255,0.2); 
                color: #fff; 
                border-radius: 12px;
                font-size: 16px;
                outline: none;
                box-sizing: border-box;
                transition: all 0.3s ease;
            }
            .form-control:focus {
                border-color: #ffcc00;
                box-shadow: 0 0 0 3px rgba(255, 204, 0, 0.2);
            }
            .form-control::placeholder { color: rgba(255,255,255,0.6); }
            .btn { 
                background: linear-gradient(135deg, #ffcc00, #ff9900);
                color: #000; 
                border: none; 
                padding: 18px 24px; 
                cursor: pointer; 
                width: 100%;
                border-radius: 12px;
                font-size: 16px;
                font-weight: 600;
                transition: all 0.3s ease;
                display: flex;
                align-items: center;
                justify-content: center;
                gap: 10px;
            }
            .btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 10px 30px rgba(255, 204, 0, 0.3);
            }
            .error {
                background: rgba(239, 68, 68, 0.2);
                color: #fecaca;
                padding: 12px 16px;
                border-radius: 8px;
                margin-bottom: 20px;
                border-left: 4px solid #ef4444;
            }
            .footer {
                text-align: center;
                margin-top: 30px;
                font-size: 0.8rem;
                opacity: 0.7;
            }
        </style>
    </head>
    <body>
        <div class="login-container">
            <div class="logo">SmallMart</div>
            <div class="version"> E-Commerce Platform v6.1.5</div>
            
            <?php if (isset($_SESSION['error'])): ?>
                <div class="error">
                    <i class="fas fa-exclamation-triangle"></i> 
                    <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                </div>
            <?php endif; ?>
            
            <form method="post" action="?">
                <div class="form-group">
                    <label class="form-label">Admin Password</label>
                    <input type="password" name="password" class="form-control" placeholder="Enter your admin password" required autofocus>
                </div>
                <input type="hidden" name="action" value="login">
                <button type="submit" class="btn">
                    <i class="fas fa-sign-in-alt"></i> 
                    Access Admin Panel
                </button>
            </form>
            
            <div class="footer">
                SmallMart v6<br>
                &copy; <?php echo COPYRIGHT_YEAR; ?> All Rights Reserved
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Get current theme
$currentThemeId = isset($settings['theme']) && !empty($settings['theme']) ? $settings['theme'] : 'professional_dark';

if (!is_array($themes) || empty($themes)) {
    $themes = $defaultThemes;
    saveData(THEMES_FILE, $themes);
}

if (!isset($themes[$currentThemeId]) || !is_array($themes[$currentThemeId])) {
    $currentThemeId = 'professional_dark';
    if (!isset($themes[$currentThemeId])) {
        $themes = array_merge($defaultThemes, $themes);
        saveData(THEMES_FILE, $themes);
    }
}

$currentTheme = $themes[$currentThemeId];

if (!isset($currentTheme['colors']) || !is_array($currentTheme['colors'])) {
    $currentTheme = $defaultThemes['professional_dark'];
}

// Filter and pagination
$search = '';
$category = '';
$mode = 'all';
$page = 1;

if (isset($_GET['q']) && is_string($_GET['q'])) {
    $search = sanitizeInput($_GET['q'], false);
}
if (isset($_GET['category']) && is_string($_GET['category'])) {
    $category = sanitizeInput($_GET['category'], false);
}
if (isset($_GET['mode']) && is_string($_GET['mode'])) {
    $mode = sanitizeInput($_GET['mode'], false);
    if (!in_array($mode, ['all', 'auction', 'buy_now', 'make_offer'])) {
        $mode = 'all';
    }
}
if (isset($_GET['page']) && is_numeric($_GET['page'])) {
    $page = max(1, intval($_GET['page']));
}

$filteredItems = is_array($items) ? $items : [];

// Apply filters
if ($search) {
    $filteredItems = array_filter($filteredItems, function($item) use ($search) {
        return stripos($item['title'], $search) !== false || 
               stripos(strip_tags($item['description']), $search) !== false ||
               stripos($item['tags'] ?? '', $search) !== false;
    });
}

if ($category) {
    $filteredItems = array_filter($filteredItems, function($item) use ($category) {
        return ($item['category'] ?? '') === $category;
    });
}

if ($mode !== 'all') {
    $filteredItems = array_filter($filteredItems, function($item) use ($mode) {
        return ($item['type'] ?? '') === $mode;
    });
}

// Pagination
$totalItems = count($filteredItems);
$itemsPerPage = $settings['items_per_page'] ?? ITEMS_PER_PAGE;
$totalPages = ceil($totalItems / $itemsPerPage);
$offset = ($page - 1) * $itemsPerPage;
$paginatedItems = array_slice($filteredItems, $offset, $itemsPerPage);

// Get current item for single view
$currentItem = null;
if (isset($_GET['view'])) {
    foreach ($items as $key => $item) {
        if ($item['id'] === $_GET['view']) {
            $currentItem = $item;
            if (!isset($_SESSION['viewed_' . $item['id']])) {
                $items[$key]['views'] = ($item['views'] ?? 0) + 1;
                $_SESSION['viewed_' . $item['id']] = true;
                saveData(ITEMS_FILE, $items);
            }
            break;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo sanitizeInput($settings['site_title'], false); ?><?php echo sanitizeInput($settings['seo_title_suffix'], false); ?></title>
    <meta name="description" content="<?php echo sanitizeInput($settings['seo_description'], false); ?>">
    <meta name="keywords" content="<?php echo sanitizeInput($settings['seo_keywords'], false); ?>">
    
    <!-- Open Graph / Social Media -->
    <meta property="og:title" content="<?php echo sanitizeInput($settings['site_title'], false); ?>">
    <meta property="og:description" content="<?php echo sanitizeInput($settings['seo_description'], false); ?>">
    <meta property="og:type" content="website">
    <meta name="twitter:card" content="summary_large_image">
    
    <!-- Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=Poppins:wght@300;400;500;600;700&family=Playfair+Display:wght@400;500;600;700&family=Source+Sans+Pro:wght@300;400;600;700&family=Roboto:wght@300;400;500;700&family=Lato:wght@300;400;700&family=Merriweather:wght@300;400;700&family=Open+Sans:wght@300;400;600;700&family=Montserrat:wght@300;400;500;600;700&family=Nunito:wght@300;400;600;700&family=Orbitron:wght@400;700;900&family=Oswald:wght@300;400;600;700&family=Raleway:wght@300;400;500;600;700&family=Crimson+Text:wght@400;600&family=Nunito+Sans:wght@300;400;600;700&family=Georgia:wght@400;700&display=swap" rel="stylesheet">
    
    <!-- Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    
    <!-- WYSIWYG Editor -->
    <?php if ($settings['enable_wysiwyg'] ?? true): ?>
    <link href="https://cdn.quilljs.com/1.3.7/quill.snow.css" rel="stylesheet">
    <script src="https://cdn.quilljs.com/1.3.7/quill.min.js"></script>
    <?php endif; ?>
    
    <style>
        /* Reset & Base */
        *, *::before, *::after { 
            box-sizing: border-box; 
            margin: 0; 
            padding: 0; 
        }
        
        /* Theme Variables */
        <?php echo generateThemeCSS($currentTheme); ?>
        
        body {
            background: var(--background);
            color: var(--text);
            font-family: var(--font-primary);
            line-height: 1.6;
            min-height: 100vh;
        }
        
        h1, h2, h3, h4, h5, h6 {
            font-family: var(--font-heading);
            color: var(--primary);
            margin: 0 0 1rem 0;
        }
        
        /* Link Styling */
        a {
            color: var(--primary);
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        a:hover {
            color: var(--secondary);
        }
        
        /* WYSIWYG Editor Styles */
        .ql-editor {
            background: white !important;
            color: #333 !important;
            border-radius: 8px;
            min-height: 200px;
        }
        
        .ql-container {
            background: white;
            border-radius: 0 0 8px 8px;
        }
        
        .ql-toolbar {
            background: #f8f9fa;
            border-radius: 8px 8px 0 0;
        }
        
        .ql-snow {
            border: 2px solid var(--border);
            border-radius: 8px;
        }
        
        .ql-snow .ql-editor h1, .ql-snow .ql-editor h2, .ql-snow .ql-editor h3 {
            color: #333 !important;
        }
        
        /* Layout */
        .container { 
            max-width: 1400px; 
            margin: 0 auto; 
            padding: 0 20px; 
        }
        
        /* Header */
        .header {
            background: var(--surface);
            backdrop-filter: blur(20px);
            border-bottom: 1px solid var(--border);
            position: sticky;
            top: 0;
            z-index: 1000;
            padding: 20px 0;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .logo {
            font-size: 2.5rem;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            user-select: none;
        }
        
        .tagline {
            font-size: 0.9rem;
            margin-top: 5px;
            color: var(--text-secondary);
            font-weight: 500;
        }
        
        .search-container {
            display: flex;
            max-width: 500px;
            width: 100%;
            position: relative;
        }
        
        .search-input {
            flex: 1;
            padding: 14px 20px;
            border: 2px solid var(--border);
            border-radius: 25px 0 0 25px;
            background: var(--surface);
            outline: none;
            color: var(--text);
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0,0,0,0.1);
        }
        
        .search-btn {
            padding: 14px 24px;
            border: 2px solid var(--border);
            border-left: none;
            border-radius: 0 25px 25px 0;
            cursor: pointer;
            background: var(--primary);
            color: white;
            transition: all 0.3s ease;
        }
        
        .search-btn:hover {
            background: var(--secondary);
        }
        
        /* Navigation */
        .nav {
            padding: 20px 0;
            background: transparent;
        }
        
        .nav-container {
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 12px;
        }
        
        .nav-link {
            background: var(--surface);
            color: var(--text);
            border: 1px solid var(--border);
            padding: 12px 24px;
            border-radius: 25px;
            text-decoration: none;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-weight: 500;
        }
        
        .nav-link:hover, .nav-link.active {
            background: var(--primary);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
        }
        
        /* Categories */
        .category-nav {
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 10px;
            margin: 20px 0;
        }
        
        .category-link {
            padding: 8px 16px;
            border-radius: 20px;
            background: var(--surface);
            color: var(--text);
            text-decoration: none;
            border: 1px solid var(--border);
            font-size: 14px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .category-link:hover, .category-link.active {
            background: var(--primary);
            color: white;
            transform: translateY(-2px);
        }
        
        /* Cards */
        .card {
            background: var(--card);
            backdrop-filter: blur(20px);
            border: 1px solid var(--border);
            border-radius: 16px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            overflow: hidden;
        }
        
        .card:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }
        
        /* Buttons */
        .btn {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
            filter: brightness(1.1);
            color: white;
        }
        
        .btn-sm { padding: 8px 16px; font-size: 12px; }
        .btn-lg { padding: 16px 32px; font-size: 16px; }
        .btn-secondary { 
            background: var(--surface); 
            color: var(--text);
            border: 1px solid var(--border);
        }
        .btn-secondary:hover {
            background: var(--border);
            color: var(--text);
        }
        .btn-danger { background: linear-gradient(135deg, #ef4444, #dc2626); }
        .btn-success { background: linear-gradient(135deg, #22c55e, #16a34a); }
        .btn-warning { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .btn-info { background: linear-gradient(135deg, #3b82f6, #2563eb); }
        
        /* Forms */
        .form-group { margin-bottom: 24px; }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--text);
            font-size: 14px;
        }
        
        .form-control {
            background: var(--surface);
            border: 2px solid var(--border);
            color: var(--text);
            padding: 12px 16px;
            border-radius: 8px;
            transition: all 0.3s ease;
            width: 100%;
            box-sizing: border-box;
            font-size: 14px;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0,0,0,0.1);
        }
        
        .form-control::placeholder {
            color: var(--text-secondary);
            opacity: 0.7;
        }
        
        /* Items Grid */
        .items-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 28px;
            margin: 30px 0;
        }
        
        .item-card {
            position: relative;
            overflow: hidden;
        }
        
        .item-image {
            height: 250px;
            position: relative;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 16px 16px 0 0;
            background: var(--surface);
        }
        
        .item-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: all 0.5s ease;
        }
        
        .item-image:hover img {
            transform: scale(1.1);
        }
        
        .item-details {
            padding: 24px;
        }
        
        .item-title {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 12px;
            line-height: 1.3;
            color: var(--text);
        }
        
        .item-description {
            color: var(--text-secondary);
            margin-bottom: 15px;
            line-height: 1.5;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .item-price {
            font-size: 1.6rem;
            font-weight: 700;
            margin-bottom: 12px;
            color: var(--primary);
        }
        
        .item-type {
            display: inline-block;
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            margin-bottom: 15px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .item-type.auction {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            color: white;
        }
        
        .item-type.buy_now {
            background: linear-gradient(135deg, #22c55e, #16a34a);
            color: white;
        }
        
        .item-type.make_offer {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
        }
        
        .item-meta {
            font-size: 0.8rem;
            color: var(--text-secondary);
            margin: 12px 0;
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .item-featured {
            position: absolute;
            top: 15px;
            left: 15px;
            background: linear-gradient(135deg, #fbbf24, #f59e0b);
            color: white;
            padding: 6px 12px;
            border-radius: 15px;
            font-size: 0.7rem;
            font-weight: 600;
            z-index: 2;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { box-shadow: 0 0 0 0 rgba(251, 191, 36, 0.7); }
            70% { box-shadow: 0 0 0 10px rgba(251, 191, 36, 0); }
            100% { box-shadow: 0 0 0 0 rgba(251, 191, 36, 0); }
        }
        
        /* Single Item View */
        .item-view-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 40px;
            margin: 30px 0;
        }
        
        .image-gallery {
            position: sticky;
            top: 100px;
        }
        
        .main-image {
            width: 100%;
            height: 450px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 16px;
            overflow: hidden;
            margin-bottom: 20px;
            background: var(--surface);
            border: 1px solid var(--border);
        }
        
        .main-image img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
        }
        
        .thumbnails {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(80px, 1fr));
            gap: 10px;
        }
        
        .thumbnail {
            width: 80px;
            height: 80px;
            cursor: pointer;
            border-radius: 8px;
            overflow: hidden;
            border: 2px solid transparent;
            transition: all 0.3s ease;
        }
        
        .thumbnail:hover, .thumbnail.active {
            border-color: var(--primary);
            transform: scale(1.05);
        }
        
        .thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        /* Admin Panel */
        .admin-panel {
            background: var(--card);
            backdrop-filter: blur(20px);
            border: 1px solid var(--border);
            padding: 32px;
            border-radius: 16px;
            margin: 20px 0;
        }
        
        .admin-nav {
            display: flex;
            margin-bottom: 32px;
            flex-wrap: wrap;
            gap: 12px;
            border-bottom: 1px solid var(--border);
            padding-bottom: 20px;
        }
        
        .admin-nav-link {
            padding: 12px 20px;
            border-radius: 10px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 500;
            background: var(--surface);
            color: var(--text);
            border: 1px solid var(--border);
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .admin-nav-link:hover, .admin-nav-link.active {
            background: var(--primary);
            color: white;
            transform: translateY(-2px);
        }
        
        /* Console Styles */
        .console-container {
            background: #1e1e1e;
            border: 1px solid #333;
            border-radius: 8px;
            margin: 20px 0;
            overflow: hidden;
        }
        
        .console-header {
            background: #333;
            color: #fff;
            padding: 10px 15px;
            font-weight: 600;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .console-content {
            max-height: 400px;
            overflow-y: auto;
            padding: 15px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            line-height: 1.4;
        }
        
        .console-line {
            color: #f8f8f2;
            margin: 2px 0;
            word-break: break-all;
        }
        
        .console-line.error {
            color: #ff6b6b;
        }
        
        .console-line.warning {
            color: #ffa500;
        }
        
        .console-line.info {
            color: #66d9ef;
        }
        
        .console-line.success {
            color: #a6e22e;
        }
        
        .system-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }
        
        .system-info-card {
            background: var(--surface);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 15px;
        }
        
        .system-info-card h5 {
            margin: 0 0 10px 0;
            color: var(--primary);
            font-size: 14px;
        }
        
        .system-info-card .value {
            font-size: 16px;
            font-weight: 600;
            color: var(--text);
        }
        
        .status-indicator {
            display: inline-block;
            width: 8px;
            height: 8px;
            border-radius: 50%;
            margin-right: 8px;
        }
        
        .status-indicator.good {
            background: #22c55e;
        }
        
        .status-indicator.warning {
            background: #f59e0b;
        }
        
        .status-indicator.error {
            background: #ef4444;
        }
        
        /* Stats */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 24px;
            margin: 24px 0;
        }
        
        .stat-card {
            background: var(--card);
            border: 1px solid var(--border);
            padding: 24px;
            border-radius: 16px;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 8px;
            color: var(--primary);
        }
        
        .stat-label {
            font-size: 0.9rem;
            color: var(--text-secondary);
        }
        
        /* Theme Builder */
        .theme-builder-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
            margin-top: 20px;
        }
        
        .color-picker-group {
            display: grid;
            grid-template-columns: 1fr 80px;
            gap: 10px;
            align-items: center;
        }
        
        .color-picker {
            width: 60px;
            height: 40px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .color-picker:hover {
            transform: scale(1.1);
        }
        
        .theme-preview {
            background: var(--card);
            border: 2px solid var(--border);
            border-radius: 12px;
            padding: 20px;
            position: sticky;
            top: 100px;
        }
        
        /* Mod System Styles */
        .mod-upload-section {
            background: var(--surface);
            border: 2px dashed var(--border);
            border-radius: 12px;
            padding: 30px;
            text-align: center;
            margin: 20px 0;
            transition: all 0.3s ease;
        }
        
        .mod-upload-section:hover {
            border-color: var(--primary);
            background: var(--card);
        }
        
        .mod-docs {
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 25px;
            margin: 20px 0;
            line-height: 1.6;
        }
        
        .mod-docs h4, .mod-docs h5, .mod-docs h6 {
            color: var(--primary);
            margin-top: 30px;
            margin-bottom: 15px;
        }
        
        .mod-docs h4 {
            font-size: 1.5rem;
            border-bottom: 2px solid var(--border);
            padding-bottom: 10px;
        }
        
        .mod-docs h5 {
            font-size: 1.3rem;
            color: var(--secondary);
        }
        
        .mod-docs h6 {
            font-size: 1.1rem;
            color: var(--text);
            font-weight: 600;
        }
        
        .mod-docs p {
            margin-bottom: 15px;
            color: var(--text);
        }
        
        .mod-docs ul, .mod-docs ol {
            margin: 15px 0;
            padding-left: 25px;
        }
        
        .mod-docs li {
            margin-bottom: 8px;
            color: var(--text);
        }
        
        .theme-export-section {
            background: var(--surface);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 25px;
            margin: 20px 0;
        }
        
        .theme-export-section h5 {
            margin-top: 0;
            color: var(--primary);
        }
        
        /* Messages */
        .message {
            padding: 16px 24px;
            margin: 16px 0;
            border-radius: 12px;
            border-left: 4px solid;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .message.success {
            background: rgba(34, 197, 94, 0.1);
            color: #16a34a;
            border-left-color: #16a34a;
        }
        
        .message.error {
            background: rgba(239, 68, 68, 0.1);
            color: #dc2626;
            border-left-color: #dc2626;
        }
        
        /* Data Tables */
        .data-table-container {
            background: var(--card);
            border-radius: 12px;
            overflow: hidden;
            border: 1px solid var(--border);
            margin: 20px 0;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th, .data-table td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid var(--border);
            vertical-align: middle;
        }
        
        .data-table th {
            background: var(--surface);
            font-weight: 600;
            color: var(--primary);
        }
        
        .data-table tr:hover {
            background: var(--surface);
        }
        
        .data-table tr:last-child td {
            border-bottom: none;
        }
        
        /* Action Buttons Group */
        .action-buttons {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        
        .action-buttons .btn {
            white-space: nowrap;
            min-width: auto;
            padding: 8px 12px;
        }
        
        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin: 40px 0;
        }
        
        .pagination a, .pagination span {
            padding: 10px 16px;
            border: 1px solid var(--border);
            border-radius: 8px;
            text-decoration: none;
            color: var(--text);
            transition: all 0.3s ease;
        }
        
        .pagination a:hover, .pagination .current {
            background: var(--primary);
            color: white;
        }
        
        /* Donation Section */
        .donation-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 16px;
            padding: 30px;
            text-align: center;
            margin: 30px 0;
            border: none;
        }
        
        .donation-card h3 {
            color: white;
            margin-bottom: 15px;
        }
        
        .donation-amount {
            font-size: 3rem;
            font-weight: 700;
            margin: 20px 0;
            color: #ffcc00;
        }
        
        /* Footer */
        footer {
            margin-top: 80px;
            padding: 60px 0 40px;
            background: var(--surface);
            backdrop-filter: blur(20px);
            border-top: 1px solid var(--border);
        }
        
        .footer-content {
            text-align: center;
            color: var(--text-secondary);
        }
        
        .footer-links {
            margin: 20px 0;
            display: flex;
            justify-content: center;
            gap: 30px;
            flex-wrap: wrap;
        }
        
        .footer-links a {
            color: var(--text-secondary);
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .footer-links a:hover {
            color: var(--primary);
        }
        
        /* Backlink Styles */
        .backlink {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            margin: 15px 0;
            padding: 8px 16px;
            background: rgba(255, 204, 0, 0.1);
            border: 1px solid rgba(255, 204, 0, 0.3);
            border-radius: 8px;
            color: var(--primary);
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .backlink:hover {
            background: var(--primary);
            color: white;
            transform: translateY(-2px);
        }
        
        /* Responsive Design */
        @media (max-width: 1024px) {
            .items-grid {
                grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            }
            .item-view-container {
                grid-template-columns: 1fr;
                gap: 30px;
            }
            .theme-builder-grid {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 0 15px;
            }
            .items-grid {
                grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
                gap: 20px;
            }
            .header-content {
                flex-direction: column;
                text-align: center;
            }
            .search-container {
                max-width: 100%;
            }
            .nav-container {
                justify-content: center;
            }
            .admin-nav {
                flex-direction: column;
            }
            .main-image {
                height: 300px;
            }
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            .action-buttons {
                flex-direction: column;
                gap: 4px;
            }
            .action-buttons .btn {
                width: 100%;
                justify-content: center;
            }
        }
        
        @media (max-width: 480px) {
            .items-grid {
                grid-template-columns: 1fr;
            }
            .logo {
                font-size: 2rem;
            }
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
        
        /* Animations */
        .fade-in {
            animation: fadeIn 0.6s ease;
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Utility Classes */
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .font-bold { font-weight: bold; }
        .text-primary { color: var(--primary); }
        .text-secondary { color: var(--text-secondary); }
        .mb-1 { margin-bottom: 0.5rem; }
        .mb-2 { margin-bottom: 1rem; }
        .mb-3 { margin-bottom: 1.5rem; }
        .mt-1 { margin-top: 0.5rem; }
        .mt-2 { margin-top: 1rem; }
        .mt-3 { margin-top: 1.5rem; }
        .hidden { display: none; }
        .flex { display: flex; }
        .flex-center { display: flex; align-items: center; justify-content: center; }
        .flex-between { display: flex; align-items: center; justify-content: space-between; }
        .w-full { width: 100%; }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="header">
        <div class="container">
            <div class="header-content">
                <div class="logo" oncontextmenu="handleRightClick(event)" title="Right-click for admin access">
                    <?php
                        $logoRel = (string)($settings['site_logo'] ?? '');
                        $useLogo = (bool)($settings['use_logo'] ?? false);
                        $logoPath = $logoRel ? (__DIR__ . '/' . ltrim($logoRel, '/')) : '';
                        if ($useLogo && $logoRel && file_exists($logoPath)) {
                            echo '<img src="' . htmlspecialchars($logoRel) . '?v=' . urlencode((string)@filemtime($logoPath)) . '" alt="' . htmlspecialchars((string)$settings['site_title']) . '" style="max-height:60px; width:auto; display:block;">';
                        } else {
                            echo sanitizeInput($settings['site_title'], false);
                        }
                    ?>
                    <div class="tagline"><?php echo sanitizeInput($settings['tagline'], false); ?></div>
                </div>
                
                <div class="search-container">
                    <form method="get" style="display: flex; width: 100%;">
                        <input type="text" name="q" class="search-input" placeholder="Search products..." value="<?php echo sanitizeInput($search, false); ?>">
                        <button type="submit" class="search-btn">
                            <i class="fas fa-search"></i>
                        </button>
                    </form>
                </div>
                
                <div style="display: flex; align-items: center; gap: 15px;">
                    <?php if ($settings['enable_wishlist'] ?? true): ?>
                    <a href="?wishlist=view" class="btn btn-secondary">
                        <i class="fas fa-heart"></i> Wishlist
                    </a>
                    <?php endif; ?>

                    <?php if (isUserLoggedIn()): $cu = currentUser(); ?>
                    <a href="?account=settings" class="btn btn-secondary">
                        <i class="fas fa-user"></i> <?php echo sanitizeInput((string)($cu['username'] ?? 'Account'), false); ?>
                    </a>
                    <form method="post" action="?action=user_logout" style="display:inline;">
                        <button type="submit" class="btn btn-secondary">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                    <?php else: ?>
                    <a href="?account=login" class="btn btn-secondary">
                        <i class="fas fa-sign-in-alt"></i> Login
                    </a>
                    <a href="?account=register" class="btn btn-secondary">
                        <i class="fas fa-user-plus"></i> Sign Up
                    </a>
                    <?php endif; ?>

                    <?php if (isAdmin()): ?>
                    <a href="?admin=dashboard" class="btn">
                        <i class="fas fa-cog"></i> Admin
                    </a>
                    <form method="post" action="?action=admin_logout" style="display:inline;">
                        <input type="hidden" name="csrf_token" value="<?php echo generateCSRF(); ?>">
                        <button type="submit" class="btn btn-secondary">
                            <i class="fas fa-sign-out-alt"></i> Admin Logout
                        </button>
                    </form>

                    <?php endif; ?>
                </div>
            </div>
        </div>
    </header>
    
    <!-- Navigation -->
    <nav class="nav">
        <div class="container">
            <div class="nav-container">
                <a href="?mode=all" class="nav-link <?php echo $mode === 'all' ? 'active' : ''; ?>">
                    <i class="fas fa-th-large"></i> All Items
                </a>
                <?php if ($settings['enabled_modes']['auction'] ?? false): ?>
                <a href="?mode=auction" class="nav-link <?php echo $mode === 'auction' ? 'active' : ''; ?>">
                    <i class="fas fa-gavel"></i> Auctions
                </a>
                <?php endif; ?>
                <?php if ($settings['enabled_modes']['buy_now'] ?? false): ?>
                <a href="?mode=buy_now" class="nav-link <?php echo $mode === 'buy_now' ? 'active' : ''; ?>">
                    <i class="fas fa-shopping-cart"></i> Buy Now
                </a>
                <?php endif; ?>
                <?php if ($settings['enabled_modes']['make_offer'] ?? false): ?>
                <a href="?mode=make_offer" class="nav-link <?php echo $mode === 'make_offer' ? 'active' : ''; ?>">
                    <i class="fas fa-handshake"></i> Make Offer
                </a>
                <?php endif; ?>
                <?php $navPages = loadPages(); $curP = strtolower((string)safeGet($_GET, 'p', '')); ?>
                <?php foreach ($navPages as $pslug => $pg): ?>
                    <?php if (!empty($pg['show_in_nav'])): ?>
                    <a href="?p=<?php echo urlencode((string)$pslug); ?>" class="nav-link <?php echo $curP === (string)$pslug ? 'active' : ''; ?>">
                        <i class="fas fa-file-alt"></i> <?php echo sanitizeInput((string)($pg['title'] ?? $pslug), false); ?>
                    </a>
                    <?php endif; ?>
                <?php endforeach; ?>

            </div>
            
            <?php if (($settings['enable_categories'] ?? false) && !empty($categories)): ?>
<!-- Category Navigation -->
            <div class="category-nav">
                <a href="?" class="category-link <?php echo empty($category) ? 'active' : ''; ?>">
                    <i class="fas fa-th"></i> All Categories
                </a>
                <?php foreach ($categories as $catId => $cat): ?>
                    <?php if ($cat['enabled'] ?? false): ?>
                    <a href="?category=<?php echo $catId; ?>" class="category-link <?php echo $category === $catId ? 'active' : ''; ?>">
                        <i class="<?php echo $cat['icon']; ?>"></i>
                        <?php echo sanitizeInput($cat['name'], false); ?>
                    </a>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
    </nav>
<?php endif; ?>

    
    <!-- Main Content -->
    <div class="container">
        <?php if (isset($_SESSION['message'])): ?>
            <div class="message success fade-in">
                <i class="fas fa-check-circle"></i> 
                <?php echo sanitizeInput($_SESSION['message'], false); unset($_SESSION['message']); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="message error fade-in">
                <i class="fas fa-exclamation-circle"></i> 
                <?php echo sanitizeInput($_SESSION['error'], false); unset($_SESSION['error']); ?>
            </div>
        <?php endif; ?>
        
        <?php
        // Main content rendering
        if (isset($_GET['account'])) {
            $accountPage = (string)$_GET['account'];
            $msg = safeGet($_SESSION, 'message'); unset($_SESSION['message']);
            $err = safeGet($_SESSION, 'error'); unset($_SESSION['error']);
            echo '<div class="container" style="padding: 30px 0;">';
            if ($msg) { echo '<div class="alert alert-success" style="margin-bottom:16px;">' . sanitizeInput($msg, true) . '</div>'; }
            if ($err) { echo '<div class="alert alert-error" style="margin-bottom:16px;">' . sanitizeInput($err, true) . '</div>'; }

            if ($accountPage === 'verify_email') {
                $token = (string)safeGet($_GET, 'token');
                $row = consumeToken($token, 'email_verify');
                if ($row && !empty($row['user_id'])) {
                    $uid = (string)$row['user_id'];
                    $users = loadUsers();
                    if (isset($users[$uid])) {
                        $users[$uid]['email_verified'] = true;
                        $users[$uid]['updated_at'] = date('c');
                        saveUsers($users);
                        echo '<div class="card"><h2>Email verified</h2><p>You can now use all account features.</p><a class="btn" href="?account=settings">Go to account settings</a></div>';
                    } else {
                        echo '<div class="card"><h2>Invalid token</h2><p>User not found.</p></div>';
                    }
                } else {
                    echo '<div class="card"><h2>Invalid or expired link</h2><p>Please request a new verification email.</p></div>';
                }
                echo '</div>';
        } elseif ($accountPage === 'reset_password') {
                $token = (string)safeGet($_GET, 'token');
                echo '<div class="card" style="max-width:520px; margin:0 auto;">';
                echo '<h2>Reset password</h2>';
                echo '<form method="post" action="?action=user_reset_password">';
                echo '<input type="hidden" name="token" value="' . htmlspecialchars($token) . '">';
                echo '<div class="form-group"><label class="form-label">New password</label><input type="password" name="new_password" class="form-control" required></div>';
                echo '<button class="btn btn-lg" type="submit"><i class="fas fa-key"></i> Reset</button>';
                echo '</form></div>';
                echo '</div>';
            } elseif ($accountPage === 'register') {
                echo '<div class="card" style="max-width:520px; margin:0 auto;">';
                echo '<h2>Create account</h2>';
                echo '<form method="post" action="?action=user_register">';
                echo '<div class="form-group"><label class="form-label">Username</label><input type="text" name="username" class="form-control" required></div>';
                echo '<div class="form-group"><label class="form-label">Email</label><input type="email" name="email" class="form-control" required></div>';
                echo '<div class="form-group"><label class="form-label">Password</label><input type="password" name="password" class="form-control" required></div>';
                echo '<button class="btn btn-lg" type="submit"><i class="fas fa-user-plus"></i> Register</button>';
                echo '<p class="muted" style="margin-top:14px;">Already have an account? <a href="?account=login">Sign in</a></p>';
                echo '</form></div>';
                echo '</div>';
            } elseif ($accountPage === 'login') {
                echo '<div class="card" style="max-width:520px; margin:0 auto;">';
                echo '<h2>Sign in</h2>';
                echo '<form method="post" action="?action=user_login">';
                echo '<div class="form-group"><label class="form-label">Username or email</label><input type="text" name="identifier" class="form-control" required></div>';
                echo '<div class="form-group"><label class="form-label">Password</label><input type="password" name="password" class="form-control" required></div>';
                echo '<button class="btn btn-lg" type="submit"><i class="fas fa-sign-in-alt"></i> Login</button>';
                echo '</form>';
                echo '<hr style="margin:18px 0; border:0; border-top:1px solid var(--border);">';
                echo '<h3 style="margin-bottom:10px;">Forgot password?</h3>';
                echo '<form method="post" action="?action=user_request_reset">';
                echo '<div class="form-group"><label class="form-label">Email</label><input type="email" name="email" class="form-control" required></div>';
                echo '<button class="btn btn-secondary" type="submit"><i class="fas fa-envelope"></i> Send reset link</button>';
                echo '</form>';
                echo '<p class="muted" style="margin-top:14px;">No account? <a href="?account=register">Create one</a></p>';
                echo '</div>';
                echo '</div>';
            } else {
                requireUser();
                $cu = currentUser();
                echo '<div class="card" style="max-width:700px; margin:0 auto;">';
                echo '<h2>Account settings</h2>';
                echo '<div class="muted" style="margin-bottom:14px;">Email verified: ' . (!empty($cu['email_verified']) ? '<b>Yes</b>' : '<b>No</b>') . '</div>';
                echo '<form method="post" action="?action=user_update_account">';
                echo '<div class="form-group"><label class="form-label">Username</label><input type="text" name="username" class="form-control" value="' . sanitizeInput((string)($cu['username'] ?? ''), false) . '" required></div>';
                echo '<div class="form-group"><label class="form-label">Email</label><input type="email" name="email" class="form-control" value="' . sanitizeInput((string)($cu['email'] ?? ''), false) . '" required></div>';
                echo '<div class="form-group"><label class="form-label">New password (optional)</label><input type="password" name="new_password" class="form-control"></div>';
                echo '<button class="btn btn-lg" type="submit"><i class="fas fa-save"></i> Save changes</button>';
                echo '</form></div>';
                echo '</div>';
            }
        } elseif (!empty($_GET['p'])) {
            $slug = strtolower((string)safeGet($_GET, 'p', ''));
            $pg = getPageBySlug($slug);
            if (!$pg) {
                echo '<div class="card"><h2>Page not found</h2><p>That page does not exist.</p></div>';
            } else {
                echo '<div class="card fade-in">';
                echo '<h2>' . sanitizeInput((string)($pg['title'] ?? $slug), false) . '</h2>';
                echo '<div class="muted" style="margin-bottom:14px;">Last updated: ' . sanitizeInput(substr((string)($pg['updated_at'] ?? $pg['created_at'] ?? ''), 0, 10), false) . '</div>';
                echo '<div class="content">' . sanitizeInput((string)($pg['content'] ?? ''), true) . '</div>';
                echo '</div>';
            }
        } elseif (isAdmin() && isset($_GET['admin'])) {
            $adminPage = $_GET['admin'];
            ?>
            <div class="admin-panel fade-in">
                <h2><i class="fas fa-cog"></i> SmallMart v6 Admin Panel</h2>
                <p style="color: var(--text-secondary); margin-bottom: 20px;"> commercial e-commerce management system with enhanced mod system and theme export/import</p>
                
                <div class="admin-nav">
                    <a href="?admin=dashboard" class="admin-nav-link <?php echo $adminPage === 'dashboard' ? 'active' : ''; ?>">
                        <i class="fas fa-tachometer-alt"></i> Dashboard
                    </a>
                    <a href="?admin=items" class="admin-nav-link <?php echo $adminPage === 'items' ? 'active' : ''; ?>">
                        <i class="fas fa-box"></i> Items (<?php echo count($items); ?>)
                    </a>
                    <a href="?admin=add_item" class="admin-nav-link <?php echo $adminPage === 'add_item' ? 'active' : ''; ?>">
                        <i class="fas fa-plus"></i> Add Item
                    </a>
                    <a href="?admin=orders" class="admin-nav-link <?php echo $adminPage === 'orders' ? 'active' : ''; ?>">
                        <i class="fas fa-shopping-cart"></i> Orders (<?php echo count($orders); ?>)
                    </a>
                    <a href="?admin=mods" class="admin-nav-link <?php echo $adminPage === 'mods' ? 'active' : ''; ?>">
                        <i class="fas fa-puzzle-piece"></i> Mods (<?php echo count(array_filter($mods, function($m) { return $m['enabled'] ?? false; })); ?>)
                    </a>
                    <a href="?admin=theme_builder" class="admin-nav-link <?php echo $adminPage === 'theme_builder' ? 'active' : ''; ?>">
                        <i class="fas fa-palette"></i> Themes (<?php echo count($themes); ?>)
                    </a>
                    <a href="?admin=console" class="admin-nav-link <?php echo $adminPage === 'console' ? 'active' : ''; ?>">
                        <i class="fas fa-terminal"></i> Console
                    </a>
                    <a href="?admin=settings" class="admin-nav-link <?php echo $adminPage === 'settings' ? 'active' : ''; ?>">
                        <i class="fas fa-cogs"></i> Settings
                    </a>

                    <a href="?admin=pages" class="admin-nav-link <?php echo $adminPage === 'pages' ? 'active' : ''; ?>">
                        <i class="fas fa-file-alt"></i> Pages
                    </a>
                    <a href="?admin=users" class="admin-nav-link <?php echo $adminPage === 'users' ? 'active' : ''; ?>">
                        <i class="fas fa-users"></i> Users
                    </a>
                    <a href="?admin=license" class="admin-nav-link <?php echo $adminPage === 'license' ? 'active' : ''; ?>">
                        <i class="fas fa-certificate"></i> License
                    </a>

                    <a href="?action=backup" class="admin-nav-link">
                        <i class="fas fa-download"></i> Backup
                    </a>
               
                </div>
                
                <?php
                switch ($adminPage) {
                    case 'console':
                        echo '<h3><i class="fas fa-terminal"></i> System Console & Monitoring</h3>';
                        echo '<p style="color: var(--text-secondary); margin-bottom: 30px;">Real-time system monitoring, logs, and debugging tools</p>';
                        
                        // System Status
                        $systemStatus = getSystemStatus();
                        echo '<h4><i class="fas fa-heartbeat"></i> System Status</h4>';
                        echo '<div class="system-info">';
                        
                        echo '<div class="system-info-card">';
                        echo '<h5><span class="status-indicator good"></span>PHP Version</h5>';
                        echo '<div class="value">' . $systemStatus['php_version'] . '</div>';
                        echo '</div>';
                        
                        echo '<div class="system-info-card">';
                        echo '<h5><span class="status-indicator good"></span>Memory Usage</h5>';
                        echo '<div class="value">' . formatBytes($systemStatus['memory_usage']) . ' / ' . $systemStatus['memory_limit'] . '</div>';
                        echo '</div>';
                        
                        echo '<div class="system-info-card">';
                        echo '<h5><span class="status-indicator ' . ($systemStatus['uploads_writable'] ? 'good' : 'error') . '"></span>Uploads</h5>';
                        echo '<div class="value">' . ($systemStatus['uploads_writable'] ? 'Writable' : 'Not Writable') . '</div>';
                        echo '</div>';
                        
                        echo '<div class="system-info-card">';
                        echo '<h5><span class="status-indicator ' . ($systemStatus['data_writable'] ? 'good' : 'error') . '"></span>Data</h5>';
                        echo '<div class="value">' . ($systemStatus['data_writable'] ? 'Writable' : 'Not Writable') . '</div>';
                        echo '</div>';
                        
                        echo '<div class="system-info-card">';
                        echo '<h5><span class="status-indicator good"></span>Server</h5>';
                        echo '<div class="value">' . substr($systemStatus['server_software'], 0, 20) . '</div>';
                        echo '</div>';
                        
                        echo '</div>';
                        
                        // Log Viewer
                        echo '<h4 style="margin-top: 40px;"><i class="fas fa-list-alt"></i> System Logs</h4>';
                        
                        $logTabs = [
                            'system' => ['System Log', SYSTEM_LOG_FILE],
                            'error' => ['Error Log', ERROR_LOG_FILE],
                            'admin' => ['Admin Log', ADMIN_LOG_FILE],
                            'mod' => ['Mod Log', MOD_LOG_FILE]
                        ];
                        
                        $activeTab = safeGet($_GET, 'log', 'system');
                        
                        echo '<div style="display: flex; gap: 10px; margin-bottom: 20px; border-bottom: 1px solid var(--border); padding-bottom: 15px;">';
                        foreach ($logTabs as $logId => $logInfo) {
                            $isActive = $activeTab === $logId;
                            echo '<a href="?admin=console&log=' . $logId . '" class="btn ' . ($isActive ? '' : 'btn-secondary') . '">' . $logInfo[0] . '</a>';
                        }
                        echo '</div>';
                        
                        if (isset($logTabs[$activeTab])) {
                            $logFile = $logTabs[$activeTab][1];
                            $logLines = Logger::readLog($logFile, 100);
                            
                            echo '<div class="console-container">';
                            echo '<div class="console-header">';
                            echo '<span>' . $logTabs[$activeTab][0] . ' (Last 100 entries)</span>';
                            echo '<div>';
                            echo '<form method="post" action="?action=clear_log" style="display: inline;">';
                            echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                            echo '<input type="hidden" name="log_type" value="' . $activeTab . '">';
                            echo '<button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Clear this log file? This cannot be undone.\')">Clear Log</button>';
                            echo '</form>';
                            echo '</div>';
                            echo '</div>';
                            echo '<div class="console-content">';
                            
                            if (!empty($logLines)) {
                                foreach ($logLines as $line) {
                                    $class = 'console-line';
                                    if (strpos($line, '[ERROR]') !== false) {
                                        $class .= ' error';
                                    } elseif (strpos($line, '[ADMIN]') !== false) {
                                        $class .= ' info';
                                    } elseif (strpos($line, '[SYSTEM]') !== false) {
                                        $class .= ' success';
                                    }
                                    echo '<div class="' . $class . '">' . htmlspecialchars($line) . '</div>';
                                }
                            } else {
                                echo '<div class="console-line">No log entries found</div>';
                            }
                            
                            echo '</div>';
                            echo '</div>';
                        }
                        
                        // Quick System Actions
                        echo '<h4 style="margin-top: 40px;"><i class="fas fa-tools"></i> Quick Actions</h4>';
                        echo '<div style="display: flex; gap: 15px; flex-wrap: wrap; margin-top: 20px;">';
                        echo '<a href="?action=backup" class="btn"><i class="fas fa-download"></i> Create Backup</a>';
                        echo '<button onclick="refreshSystemInfo()" class="btn btn-info"><i class="fas fa-sync"></i> Refresh Status</button>';
                        echo '<button onclick="showPHPInfo()" class="btn btn-secondary"><i class="fas fa-info-circle"></i> PHP Info</button>';
                        echo '</div>';
                        break;
                        
                    case 'theme_builder':
                        echo '<h3><i class="fas fa-palette"></i> Enhanced Theme Builder (' . count($defaultThemes) . ' Built-in Themes)</h3>';
                        echo '<p style="color: var(--text-secondary); margin-bottom: 30px;">Create, export, and import custom themes with live preview. Current: <strong>' . ($themes[$settings['theme']]['name'] ?? 'Unknown') . '</strong></p>';
                        
                        // Tab Navigation
                        $themeTab = safeGet($_GET, 'tab', 'builder');
                        echo '<div style="display: flex; gap: 10px; margin-bottom: 30px; border-bottom: 1px solid var(--border); padding-bottom: 15px;">';
                        echo '<a href="?admin=theme_builder&tab=builder" class="btn ' . ($themeTab === 'builder' ? '' : 'btn-secondary') . '">Theme Builder</a>';
                        echo '<a href="?admin=theme_builder&tab=export" class="btn ' . ($themeTab === 'export' ? '' : 'btn-secondary') . '">Export/Import</a>';
                        echo '<a href="?admin=theme_builder&tab=manage" class="btn ' . ($themeTab === 'manage' ? '' : 'btn-secondary') . '">Manage Themes</a>';
                        echo '</div>';
                        
                        switch ($themeTab) {
                            case 'export':
                                echo '<h4><i class="fas fa-download"></i> Export & Import Themes</h4>';
                                
                                // Export Section
                                echo '<div class="theme-export-section">';
                                echo '<h5>Export Theme</h5>';
                                echo '<p>Export any theme to share with other SmallMart installations:</p>';
                                echo '<div style="display: flex; gap: 10px; align-items: center; margin: 15px 0;">';
                                echo '<select id="export-theme-select" class="form-control" style="max-width: 300px;">';
                                foreach ($themes as $themeId => $theme) {
                                    echo '<option value="' . $themeId . '">' . sanitizeInput($theme['name'], false) . '</option>';
                                }
                                echo '</select>';
                                echo '<a href="?action=export_theme&id=professional_dark" id="export-btn" class="btn"><i class="fas fa-download"></i> Export Theme</a>';
                                echo '</div>';
                                echo '</div>';
                                
                                // Import Section
                                echo '<div class="theme-export-section">';
                                echo '<h5>Import Theme</h5>';
                                echo '<p>Import a theme file from another SmallMart installation:</p>';
                                echo '<form method="post" action="?action=import_theme">';
                                echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                                echo '<div class="form-group">';
                                echo '<label class="form-label">Theme Data (paste exported theme code)</label>';
                                echo '<textarea name="theme_data" class="form-control" rows="8" placeholder="Paste exported theme data here..." required></textarea>';
                                echo '</div>';
                                echo '<button type="submit" class="btn"><i class="fas fa-upload"></i> Import Theme</button>';
                                echo '</form>';
                                echo '</div>';
                                break;
                                
                            case 'manage':
                                echo '<h4><i class="fas fa-list"></i> Available Themes (' . count($themes) . ')</h4>';
                                echo '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px; margin-top: 20px;">';
                                
                                foreach ($themes as $themeId => $theme) {
                                    $isActive = $settings['theme'] === $themeId;
                                    $isDefault = $theme['is_default'] ?? false;
                                    
                                    echo '<div class="card" style="padding: 20px; ' . ($isActive ? 'border: 2px solid var(--primary);' : '') . '">';
                                    echo '<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">';
                                    echo '<h5>' . sanitizeInput($theme['name'], false) . '</h5>';
                                    echo '<div>';
                                    if ($isActive) {
                                        echo '<span class="btn btn-sm" style="background: var(--primary); color: white; margin-right: 5px;">Active</span>';
                                    }
                                    if ($isDefault) {
                                        echo '<span class="btn btn-sm" style="background: var(--accent); color: white;">Built-in</span>';
                                    } elseif ($theme['is_imported'] ?? false) {
                                        echo '<span class="btn btn-sm" style="background: var(--secondary); color: white;">Imported</span>';
                                    } elseif ($theme['is_custom'] ?? false) {
                                        echo '<span class="btn btn-sm" style="background: var(--info); color: white;">Custom</span>';
                                    }
                                    echo '</div>';
                                    echo '</div>';
                                    
                                    // Color swatches
                                    echo '<div style="display: flex; gap: 8px; margin-bottom: 15px;">';
                                    if (isset($theme['colors']) && is_array($theme['colors'])) {
                                        foreach (['primary', 'secondary', 'accent'] as $colorKey) {
                                            if (isset($theme['colors'][$colorKey])) {
                                                $color = $theme['colors'][$colorKey];
                                                // Handle gradients and rgba values by using a fallback color
                                                if (strpos($color, 'rgba') !== false || strpos($color, 'linear-gradient') !== false) {
                                                    $color = '#6366f1';
                                                }
                                                echo '<div style="width: 30px; height: 30px; border-radius: 50%; background: ' . htmlspecialchars($color) . '; border: 2px solid var(--border);"></div>';
                                            }
                                        }
                                    }
                                    echo '</div>';
                                    
                                    if (isset($theme['created'])) {
                                        echo '<div style="font-size: 0.8rem; color: var(--text-secondary); margin-bottom: 15px;">Created: ' . date('M j, Y', strtotime($theme['created'])) . '</div>';
                                    }
                                    
                                    echo '<div style="display: flex; gap: 10px; flex-wrap: wrap;">';
                                    
                                    if (!$isActive) {
                                        echo '<form method="post" action="?action=save_settings" style="flex: 1;">';
                                        echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                                        echo '<input type="hidden" name="theme" value="' . htmlspecialchars($themeId) . '">';
                                        echo '<input type="hidden" name="change_theme_only" value="1">';
                                        echo '<button type="submit" class="btn btn-sm" style="width: 100%;">Activate</button>';
                                        echo '</form>';
                                    } else {
                                        echo '<div class="btn btn-sm" style="background: var(--primary); color: white; flex: 1; text-align: center;">Current</div>';
                                    }
                                    
                                    echo '<a href="?action=export_theme&id=' . htmlspecialchars($themeId) . '" class="btn btn-sm btn-info" title="Export"><i class="fas fa-download"></i></a>';
                                    
                                    if ($theme['is_custom'] ?? false) {
                                        echo '<a href="?action=delete_theme&id=' . htmlspecialchars($themeId) . '" class="btn btn-sm btn-danger" onclick="return confirm(\'Delete this custom theme?\')" title="Delete"><i class="fas fa-trash"></i></a>';
                                    }
                                    
                                    echo '</div>';
                                    echo '</div>';
                                }
                                echo '</div>';
                                break;
                                
                            default: // builder
                                // Theme Builder Form
                                echo '<div class="theme-builder-grid">';
                                
                                // Left side - Theme Builder Form
                                echo '<div>';
                                echo '<h4><i class="fas fa-paint-brush"></i> Create New Theme</h4>';
                                echo '<form method="post" action="?action=save_theme" id="theme-builder-form">';
                                echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                                
                                echo '<div class="form-group">';
                                echo '<label class="form-label">Theme Name</label>';
                                echo '<input type="text" name="theme_name" class="form-control" placeholder="My Custom Theme" required>';
                                echo '</div>';
                                
                                // Color Picker Section
                                echo '<h5 style="margin: 30px 0 20px 0;"><i class="fas fa-palette"></i> Colors</h5>';
                                
                                $colorFields = [
                                    'primary' => ['Primary Color', '#ffcc00'],
                                    'secondary' => ['Secondary Color', '#ff9900'],
                                    'accent' => ['Accent Color', '#ffd700'],
                                    'background' => ['Background Color', '#1a1a1a'],
                                    'surface' => ['Surface Color', '#2d2d2d'],
                                    'card' => ['Card Color', '#3c3c3c'],
                                    'text' => ['Text Color', '#ffffff'],
                                    'text_secondary' => ['Secondary Text', '#e0e0e0'],
                                    'border' => ['Border Color', '#ffcc0033']
                                ];
                                
                                foreach ($colorFields as $field => $data) {
                                    echo '<div class="form-group">';
                                    echo '<label class="form-label">' . $data[0] . '</label>';
                                    echo '<div class="color-picker-group">';
                                    echo '<input type="text" name="' . $field . '" class="form-control color-input" value="' . $data[1] . '" onchange="updatePreview()">';
                                    echo '<input type="color" class="color-picker" value="' . (strpos($data[1], '#') === 0 && strlen($data[1]) === 7 ? $data[1] : '#ffcc00') . '" onchange="updateColorFromPicker(this, \'' . $field . '\')">';
                                    echo '</div>';
                                    echo '</div>';
                                }
                                
                                // Font Selection
                                echo '<h5 style="margin: 30px 0 20px 0;"><i class="fas fa-font"></i> Typography</h5>';
                                
                                $fonts = [
                                    "'Inter', sans-serif" => 'Inter',
                                    "'Poppins', sans-serif" => 'Poppins',
                                    "'Roboto', sans-serif" => 'Roboto',
                                    "'Open Sans', sans-serif" => 'Open Sans',
                                    "'Lato', sans-serif" => 'Lato',
                                    "'Montserrat', sans-serif" => 'Montserrat',
                                    "'Nunito', sans-serif" => 'Nunito',
                                    "'Source Sans Pro', sans-serif" => 'Source Sans Pro',
                                    "'Playfair Display', serif" => 'Playfair Display',
                                    "'Merriweather', serif" => 'Merriweather'
                                ];
                                
                                echo '<div class="form-group">';
                                echo '<label class="form-label">Primary Font</label>';
                                echo '<select name="font_primary" class="form-control" onchange="updatePreview()">';
                                foreach ($fonts as $value => $label) {
                                    echo '<option value="' . $value . '">' . $label . '</option>';
                                }
                                echo '</select>';
                                echo '</div>';
                                
                                echo '<div class="form-group">';
                                echo '<label class="form-label">Heading Font</label>';
                                echo '<select name="font_heading" class="form-control" onchange="updatePreview()">';
                                foreach ($fonts as $value => $label) {
                                    echo '<option value="' . $value . '">' . $label . '</option>';
                                }
                                echo '</select>';
                                echo '</div>';
                                
                                echo '<button type="submit" class="btn btn-lg" style="width: 100%; margin-top: 20px;">';
                                echo '<i class="fas fa-save"></i> Save Custom Theme';
                                echo '</button>';
                                echo '</form>';
                                echo '</div>';
                                
                                // Right side - Live Preview
                                echo '<div>';
                                echo '<h4><i class="fas fa-eye"></i> Live Preview</h4>';
                                echo '<div class="theme-preview" id="theme-preview">';
                                echo '<div style="padding: 20px; background: var(--background); color: var(--text); border-radius: 8px;">';
                                echo '<h3 style="color: var(--primary); font-family: var(--font-heading);">Sample Heading</h3>';
                                echo '<p style="font-family: var(--font-primary); color: var(--text-secondary);">This is sample text to show how your theme will look.</p>';
                                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px; margin: 10px 0; border: 1px solid var(--border);">';
                                echo '<strong style="color: var(--primary);">Sample Item Title</strong><br>';
                                echo '<span style="color: var(--text-secondary);">Sample description text here</span><br>';
                                echo '<strong style="color: var(--accent);">$99.99</strong>';
                                echo '</div>';
                                echo '<button class="btn" style="margin-top: 10px;">Sample Button</button>';
                                echo '</div>';
                                echo '</div>';
                                echo '</div>';
                                
                                echo '</div>';
                                break;
                        }
                        break;
                        
                    
                    
                    case 'pages':
                        echo '<h3><i class="fas fa-file-alt"></i> Pages</h3>';
                        echo '<p style="color: var(--text-secondary); margin-bottom: 20px;">Create and manage custom pages like Terms, About, Privacy, FAQ, etc.</p>';

                        $pagesArr = loadPages();
                        if (!is_array($pagesArr)) $pagesArr = [];
                        ksort($pagesArr);

                        $editSlug = strtolower((string)safeGet($_GET, 'edit', ''));
                        $editing = $editSlug !== '';
                        $editingPage = $editing ? ($pagesArr[$editSlug] ?? null) : null;

                        echo '<div class="card" style="margin-bottom:20px;">';
                        echo '<div style="display:flex; align-items:center; justify-content:space-between; gap:12px; flex-wrap:wrap;">';
                        echo '<h4 style="margin:0;">' . ($editing ? 'Edit Page' : 'All Pages') . '</h4>';
                        echo '<a class="btn btn-secondary" href="?admin=pages&edit="><i class="fas fa-plus"></i> Add / Edit</a>';
                        echo '</div>';
                        echo '</div>';

                        // Edit/Create form
                        echo '<div class="card" style="margin-bottom:20px;">';
                        echo '<form method="post" action="?action=admin_save_page">';
                        echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                        echo '<div class="settings-grid">';
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Slug (URL)</label>';
                        echo '<input type="text" name="slug" class="form-control" value="' . sanitizeInput((string)($editingPage['slug'] ?? $editSlug), false) . '" placeholder="about" required>';
                        echo '<div class="muted" style="margin-top:6px;">Lowercase letters, numbers, hyphens only. Page URL: <code>?p=your-slug</code></div>';
                        echo '</div>';
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Title</label>';
                        echo '<input type="text" name="title" class="form-control" value="' . sanitizeInput((string)($editingPage['title'] ?? ''), false) . '" placeholder="About Us" required>';
                        echo '</div>';
                        echo '</div>';

                        $checkedNav = !empty($editingPage['show_in_nav']) ? 'checked' : '';
                        echo '<label style="display:flex; align-items:center; gap:10px; margin: 10px 0; cursor:pointer;">';
                        echo '<input type="checkbox" name="show_in_nav" ' . $checkedNav . '>';
                        echo '<i class="fas fa-link"></i> Show in top navigation';
                        echo '</label>';

                        $checkedFooter = !empty($editingPage['show_in_footer']) ? 'checked' : '';
                        echo '<label style="display:flex; align-items:center; gap:10px; margin: 10px 0; cursor:pointer;">';
                        echo '<input type="checkbox" name="show_in_footer" ' . $checkedFooter . '>';
                        echo '<i class="fas fa-shoe-prints"></i> Show in footer';
                        echo '</label>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Content (HTML allowed)</label>';
                        echo '<textarea name="content" class="form-control" rows="10" style="min-height:240px;">' . sanitizeInput((string)($editingPage['content'] ?? ''), true) . '</textarea>';
                        echo '<div class="muted" style="margin-top:6px;">Allowed tags: p, br, strong, em, u, ol/ul/li, headings, a, img, blockquote, code.</div>';
                        echo '</div>';

                        echo '<button class="btn btn-lg" type="submit"><i class="fas fa-save"></i> Save Page</button>';
                        echo '</form>';
                        echo '</div>';

                        // List table
                        echo '<div class="card">';
                        echo '<div style="overflow-x:auto;">';
                        echo '<table class="table">';
                        echo '<thead><tr><th>Slug</th><th>Title</th><th>In Nav</th><th>Updated</th><th style="text-align:right;">Actions</th></tr></thead><tbody>';
                        foreach ($pagesArr as $slug => $pg) {
                            echo '<tr>';
                            echo '<td><code>' . sanitizeInput((string)$slug, false) . '</code></td>';
                            echo '<td><strong>' . sanitizeInput((string)($pg['title'] ?? ''), false) . '</strong></td>';
                            echo '<td>' . (!empty($pg['show_in_nav']) ? '<span class="badge badge-success">Yes</span>' : '<span class="badge badge-warning">No</span>') . '</td>';
                            echo '<td style="white-space:nowrap;">' . sanitizeInput(substr((string)($pg['updated_at'] ?? $pg['created_at'] ?? ''), 0, 10), false) . '</td>';
                            echo '<td style="text-align:right; white-space:nowrap;">';
                            echo '<a class="btn btn-secondary btn-sm" href="?admin=pages&edit=' . urlencode((string)$slug) . '"><i class="fas fa-edit"></i> Edit</a> ';
                            echo '<a class="btn btn-secondary btn-sm" href="?p=' . urlencode((string)$slug) . '" target="_blank"><i class="fas fa-external-link-alt"></i> View</a> ';
                            echo '<form method="post" action="?action=admin_delete_page" style="display:inline-block; margin-left:6px;" onsubmit="return confirm(\'Delete this page?\');">';
                            echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                            echo '<input type="hidden" name="slug" value="' . htmlspecialchars((string)$slug) . '">';
                            echo '<button class="btn btn-secondary btn-sm" type="submit"><i class="fas fa-trash"></i> Delete</button>';
                            echo '</form>';
                            echo '</td>';
                            echo '</tr>';
                        }
                        echo '</tbody></table>';
                        echo '</div>';
                        echo '</div>';
                        break;

case 'users':
                        echo '<h3><i class="fas fa-users"></i> User Accounts</h3>';
                        echo '<p style="color: var(--text-secondary); margin-bottom: 20px;">Manage customer accounts, roles, and access.</p>';
                        $usersArr = loadUsers();
                        if (!is_array($usersArr)) $usersArr = [];
                        echo '<div class="card">';
                        echo '<div style="overflow-x:auto;">';
                        echo '<table class="table">';
                        echo '<thead><tr><th>Username</th><th>Email</th><th>Role</th><th>Verified</th><th>Status</th><th>Created</th><th style="text-align:right;">Actions</th></tr></thead><tbody>';
                        foreach ($usersArr as $uid => $u) {
                            $banned = !empty($u['banned']);
                            echo '<tr>';
                            echo '<td><strong>' . sanitizeInput((string)($u['username'] ?? ''), false) . '</strong></td>';
                            echo '<td>' . sanitizeInput((string)($u['email'] ?? ''), false) . '</td>';
                            echo '<td><span class="badge ' . ((($u['role'] ?? '') === 'admin') ? 'badge-success' : 'badge-info') . '">' . sanitizeInput((string)($u['role'] ?? 'user'), false) . '</span></td>';
                            echo '<td>' . (!empty($u['email_verified']) ? '<span class="badge badge-success">Yes</span>' : '<span class="badge badge-warning">No</span>') . '</td>';
                            echo '<td>' . ($banned ? '<span class="badge badge-danger">Banned</span>' : '<span class="badge badge-success">Active</span>') . '</td>';
                            echo '<td style="white-space:nowrap;">' . sanitizeInput(substr((string)($u['created_at'] ?? ''), 0, 10), false) . '</td>';
                            echo '<td style="text-align:right; white-space:nowrap;">';
                            echo '<form method="post" action="?action=admin_user_update" style="display:inline-block; margin-left:6px;">';
                            echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                            echo '<input type="hidden" name="user_id" value="' . htmlspecialchars((string)$uid) . '">';
                            echo '<input type="hidden" name="op" value="toggle_ban">';
                            echo '<button class="btn btn-secondary btn-sm" type="submit"><i class="fas fa-ban"></i> ' . ($banned ? 'Unban' : 'Ban') . '</button>';
                            echo '</form>';
                            echo '<form method="post" action="?action=admin_user_update" style="display:inline-block; margin-left:6px;">';
                            echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                            echo '<input type="hidden" name="user_id" value="' . htmlspecialchars((string)$uid) . '">';
                            echo '<input type="hidden" name="op" value="toggle_admin">';
                            echo '<button class="btn btn-secondary btn-sm" type="submit"><i class="fas fa-user-shield"></i> Toggle Admin</button>';
                            echo '</form>';
                            echo '<form method="post" action="?action=admin_user_update" style="display:inline-block; margin-left:6px;">';
                            echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                            echo '<input type="hidden" name="user_id" value="' . htmlspecialchars((string)$uid) . '">';
                            echo '<input type="hidden" name="op" value="reset_pass">';
                            echo '<button class="btn btn-secondary btn-sm" type="submit"><i class="fas fa-key"></i> Reset Pass</button>';
                            echo '</form>';
                            echo '</td>';
                            echo '</tr>';
                        }
                        echo '</tbody></table>';
                        echo '</div>';
                        echo '</div>';
                        break;

                    case 'license':
                        echo '<h3><i class="fas fa-certificate"></i> License</h3>';
                        echo '<p style="color: var(--text-secondary); margin-bottom: 20px;">Backlink removal license (SmallMart.Site footer link).</p>';

                        $lic = loadData(LICENSE_FILE);
                        $paid = isDonationComplete();
                        echo '<div class="card">';
                        echo '<h4>Status</h4>';
                        echo $paid ? '<div class="alert alert-success">License: <b>PAID</b></div>' : '<div class="alert alert-warning">License: <b>NOT PAID</b> — backlink is enforced.</div>';
                        if ($paid && is_array($lic)) {
                            echo '<div class="muted">Paid at: ' . sanitizeInput((string)($lic['paid_at'] ?? ''), false) . '</div>';
                            echo '<div class="muted">Provider: ' . sanitizeInput((string)($lic['provider'] ?? ''), false) . '</div>';
                            echo '<div class="muted">Txn: ' . sanitizeInput((string)($lic['provider_txn'] ?? ''), false) . '</div>';
                        }

                        echo '<hr style="margin:16px 0; border:0; border-top:1px solid var(--border);">';
                        echo '<h4>Purchase</h4>';
                        echo '<p class="muted">Pay <b>$' . number_format(BACKLINK_REMOVAL_PRICE, 2) . '</b> to ' . BACKLINK_REMOVAL_PAYPAL_EMAIL . ' via PayPal. After PayPal confirms (IPN), the backlink toggle becomes available in Settings.</p>';

                        $custom = paypalCustomForLicense();
                        $notify = baseUrl() . '?paypal_ipn=1';
                        $return = baseUrl() . '?admin=license';
                        $cancel = baseUrl() . '?admin=license';

                        $paypalUrl = (!empty($settings['paypal_sandbox']) ? 'https://www.sandbox.paypal.com/cgi-bin/webscr' : 'https://www.paypal.com/cgi-bin/webscr');
                        echo '<form method="post" action="' . htmlspecialchars($paypalUrl) . '">';
                        echo '<input type="hidden" name="cmd" value="_xclick">';
                        echo '<input type="hidden" name="business" value="' . htmlspecialchars(BACKLINK_REMOVAL_PAYPAL_EMAIL) . '">';
                        echo '<input type="hidden" name="item_name" value="SmallMart Backlink Removal License">';
                        echo '<input type="hidden" name="amount" value="' . htmlspecialchars((string)BACKLINK_REMOVAL_PRICE) . '">';
                        echo '<input type="hidden" name="currency_code" value="USD">';
                        echo '<input type="hidden" name="no_note" value="1">';
                        echo '<input type="hidden" name="notify_url" value="' . htmlspecialchars($notify) . '">';
                        echo '<input type="hidden" name="return" value="' . htmlspecialchars($return) . '">';
                        echo '<input type="hidden" name="cancel_return" value="' . htmlspecialchars($cancel) . '">';
                        echo '<input type="hidden" name="custom" value="' . htmlspecialchars($custom) . '">';
                        echo '<button class="btn btn-lg" type="submit"><i class="fab fa-paypal"></i> Pay with PayPal</button>';
                        echo '</form>';

                        echo '<div class="muted" style="margin-top:12px;">IPN endpoint: <code>' . htmlspecialchars($notify) . '</code></div>';
                        echo '</div>';
                        break;

case 'settings':
                        echo '<h3><i class="fas fa-cogs"></i> System Settings</h3>';
                        echo '<form method="post" action="?action=save_settings" enctype="multipart/form-data">';
                        echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                        
                        echo '<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 40px;">';
                        
                        // Left column
                        echo '<div>';
                        echo '<h4>Site Configuration</h4>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Site Title</label>';
                        echo '<input type="text" name="site_title" class="form-control" value="' . sanitizeInput($settings['site_title'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Admin Username</label>';
                        echo '<input type="text" name="admin_username" class="form-control" value="' . sanitizeInput($settings['admin_username'] ?? 'admin', false) . '">';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Admin Email</label>';
                        echo '<input type="email" name="admin_email" class="form-control" value="' . sanitizeInput($settings['admin_email'] ?? '', false) . '">';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Tagline</label>';
                        echo '<input type="text" name="tagline" class="form-control" value="' . sanitizeInput($settings['tagline'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Site Logo (optional)</label>';
                        echo '<input type="file" name="site_logo_file" class="form-control" accept="image/png,image/jpeg,image/webp">';
                        $curLogo = (string)($settings['site_logo'] ?? '');
                        if ($curLogo) {
                            echo '<div class="muted" style="margin-top:6px;">Current:</div>';
                            echo '<div style="padding:8px; border:1px solid var(--border); border-radius:10px; display:inline-block; background:rgba(255,255,255,0.03);">';
                            echo '<img src="' . htmlspecialchars($curLogo) . '?v=' . urlencode((string)@filemtime(__DIR__ . '/' . ltrim($curLogo,'/'))) . '" style="max-height:50px; width:auto;">';
                            echo '</div>';
                        }
                        echo '<div class="muted" style="margin-top:6px;">Auto-resized to fit 320×80.</div>';
                        echo '</div>';

                        echo '<div class="form-group">';
                        $checkedLogo = !empty($settings['use_logo']) ? 'checked' : '';
                        echo '<label class="form-label">Use Logo Instead of Text</label>';
                        echo '<label class="toggle">';
                        echo '<input type="checkbox" name="use_logo" value="1" ' . $checkedLogo . '>';
                        echo '<span class="toggle-slider"></span>';
                        echo '</label>';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Active Theme</label>';
                        echo '<select name="theme" class="form-control">';
                        foreach ($themes as $themeId => $theme) {
                            $selected = ($settings['theme'] === $themeId) ? 'selected' : '';
                            $themeName = isset($theme['name']) ? sanitizeInput($theme['name'], false) : $themeId;
                            echo '<option value="' . htmlspecialchars($themeId) . '" ' . $selected . '>' . $themeName . '</option>';
                        }
                        echo '</select>';
                        echo '<small style="color: var(--text-secondary);">Visit Theme Builder for more theme options and customization</small>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Currency Symbol</label>';
                        echo '<input type="text" name="currency" class="form-control" value="' . sanitizeInput($settings['currency'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Timezone</label>';
                        echo '<select name="timezone" class="form-control">';
                        $timezones = [
                            'America/New_York' => 'Eastern Time',
                            'America/Chicago' => 'Central Time', 
                            'America/Denver' => 'Mountain Time',
                            'America/Los_Angeles' => 'Pacific Time',
                            'UTC' => 'UTC',
                            'Europe/London' => 'GMT',
                            'Europe/Paris' => 'Central European Time',
                            'Asia/Tokyo' => 'Japan Time',
                            'Australia/Sydney' => 'Australian Eastern Time'
                        ];
                        foreach ($timezones as $tz => $label) {
                            $selected = ($settings['timezone'] ?? 'America/New_York') === $tz ? 'selected' : '';
                            echo '<option value="' . $tz . '" ' . $selected . '>' . $label . '</option>';
                        }
                        echo '</select>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Items Per Page</label>';
                        echo '<input type="number" name="items_per_page" class="form-control" value="' . ($settings['items_per_page'] ?? ITEMS_PER_PAGE) . '" min="1" max="100">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Max Images Per Item</label>';
                        echo '<input type="number" name="max_images" class="form-control" value="' . ($settings['max_images'] ?? MAX_IMAGES_PER_ITEM) . '" min="1" max="100">';
                        echo '</div>';
                        
                        echo '<h4>PayPal Payment Settings</h4>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">PayPal Email Address</label>';
                        echo '<input type="email" name="paypal_email" class="form-control" value="' . sanitizeInput($settings['mods']['paypal_integration']['settings']['paypal_email'] ?? '', false) . '" placeholder="your-paypal@email.com">';
                        echo '<small style="color: var(--text-secondary);">Enter your PayPal email to receive payments</small>';
                        echo '</div>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 15px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="paypal_sandbox_mode" ' . (($settings['mods']['paypal_integration']['settings']['sandbox_mode'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-flask"></i> Sandbox Mode (for testing)';
                        echo '</label>';
                        
                        echo '<h4 style="margin-top:25px;">Stripe Payment Settings</h4>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Stripe Public Key</label>';
                        echo '<input type="text" name="stripe_public_key" class="form-control" value="' . sanitizeInput($settings['stripe_public_key'] ?? '', false) . '" placeholder="pk_live_...">';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Stripe Secret Key</label>';
                        echo '<input type="password" name="stripe_secret_key" class="form-control" value="' . sanitizeInput($settings['stripe_secret_key'] ?? '', false) . '" placeholder="sk_live_...">';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">Stripe Webhook Secret</label>';
                        echo '<input type="password" name="stripe_webhook_secret" class="form-control" value="' . sanitizeInput($settings['stripe_webhook_secret'] ?? '', false) . '" placeholder="whsec_...">';
                        echo '<div class="muted" style="margin-top:6px;">Set your Stripe webhook to: <code>' . htmlspecialchars(baseUrl() . '?stripe_webhook=1') . '</code></div>';
                        echo '</div>';

                        echo '<h4 style="margin-top:25px;">Email Sender Settings</h4>';
                        echo '<div class="form-group">';
                        echo '<label class="form-label">From Email</label>';
                        echo '<input type="email" name="mail_from_email" class="form-control" value="' . sanitizeInput($settings['mail_from_email'] ?? '', false) . '" placeholder="no-reply@yourdomain.com">';
                        echo '</div>';

                        echo '<div class="form-group">';
                        echo '<label class="form-label">From Name</label>';
                        echo '<input type="text" name="mail_from_name" class="form-control" value="' . sanitizeInput($settings['mail_from_name'] ?? 'SmallMart', false) . '">';
                        echo '</div>';

                        echo '<label style="display:flex; align-items:center; gap:10px; margin: 10px 0; cursor:pointer;">';
                        echo '<input type="checkbox" name="dev_show_email_links" ' . (!empty($settings['dev_show_email_links']) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-code"></i> Dev: Show email links on-screen (for testing)';
                        echo '</label>';

                        echo '</div>';
                        
                        // Right column
                        echo '<div>';
                        echo '<h4>Features & Functionality</h4>';
                        
                        $features = [
                            'enable_wysiwyg' => 'WYSIWYG Editor',
                            'enable_reviews' => 'Customer Reviews',
                            'enable_wishlist' => 'Wishlist System',
                            'enable_categories' => 'Categories Navigation',
                            'require_login_for_purchase' => 'Require Account for Purchases',
                            'maintenance_mode' => 'Maintenance Mode'
                        ];
                        
                        foreach ($features as $key => $label) {
                            echo '<label style="display: flex; align-items: center; gap: 10px; margin: 15px 0; cursor: pointer;">';
                            echo '<input type="checkbox" name="' . $key . '" ' . (($settings[$key] ?? false) ? 'checked' : '') . '>';
                            echo $label;
                            echo '</label>';
                        }
                        
                        echo '<h4>Selling Modes</h4>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="mode_auction" ' . (($settings['enabled_modes']['auction'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-gavel"></i> Auctions';
                        echo '</label>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="mode_buy_now" ' . (($settings['enabled_modes']['buy_now'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-shopping-cart"></i> Buy It Now';
                        echo '</label>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="mode_make_offer" ' . (($settings['enabled_modes']['make_offer'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-handshake"></i> Make Offer';
                        echo '</label>';
                        
                        echo '<h4>Payment Methods</h4>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="payment_paypal" ' . (($settings['payment_methods']['paypal'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fab fa-paypal"></i> PayPal';
                        echo '</label>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="payment_stripe" ' . (($settings['payment_methods']['stripe'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fab fa-cc-stripe"></i> Stripe';
                        echo '</label>';

                        
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin: 10px 0; cursor: pointer;">';
                        echo '<input type="checkbox" name="payment_contact" ' . (($settings['payment_methods']['contact'] ?? false) ? 'checked' : '') . '>';
                        echo '<i class="fas fa-envelope"></i> Contact Seller';
                        echo '</label>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Contact Email</label>';
                        echo '<input type="email" name="contact_email" class="form-control" value="' . sanitizeInput($settings['contact_email'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">New Admin Password</label>';
                        echo '<input type="password" name="new_password" class="form-control" placeholder="Leave blank to keep current">';
                        echo '</div>';
                        
                        echo '</div>';
                        echo '</div>';
                        
                        echo '<h4>SEO & Footer Settings</h4>';
                        echo '<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">SEO Title Suffix</label>';
                        echo '<input type="text" name="seo_title_suffix" class="form-control" value="' . sanitizeInput($settings['seo_title_suffix'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Meta Description</label>';
                        echo '<textarea name="seo_description" class="form-control" rows="3">' . sanitizeInput($settings['seo_description'], false) . '</textarea>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Meta Keywords</label>';
                        echo '<input type="text" name="seo_keywords" class="form-control" value="' . sanitizeInput($settings['seo_keywords'], false) . '">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Footer Text</label>';
                        echo '<textarea name="footer_text" class="form-control" rows="3">' . sanitizeInput($settings['footer_text'], false) . '</textarea>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Footer Backlink (SmallMart.Site)</label>';
                        $paid = isDonationComplete();
                        $checked = !empty($settings['backlink_show']) ? 'checked' : '';
                        $disabled = $paid ? '' : 'disabled';
                        echo '<div class="muted" style="margin-bottom:6px;">' . ($paid ? 'License paid: you may toggle this.' : 'License not paid: backlink is enforced. Pay $' . number_format(BACKLINK_REMOVAL_PRICE, 2) . ' in Admin → License to enable toggle.') . '</div>';
                        echo '<label class="toggle">';
                        echo '<input type="checkbox" name="backlink_show" value="1" ' . $checked . ' ' . $disabled . '>';
                        echo '<span class="toggle-slider"></span>';
                        echo '</label>';
                        echo '</div>';

                        echo '</div>';
                        
                         echo '<button type="submit" class="btn btn-lg" style="margin-top: 30px;">';
                        echo '<i class="fas fa-save"></i> Save All Settings';
                        echo '</button>';
                        echo '</form>';
                        break;
                        
                    case 'dashboard':
                        echo '<h3><i class="fas fa-tachometer-alt"></i> Dashboard Overview</h3>';
                        echo '<div class="stats-grid">';
                        
                        $statsData = [
                            ['Total Items', count($items), 'fas fa-box'],
                            ['Total Views', number_format($stats['total_views'] ?? 0), 'fas fa-eye'],
                            ['Total Orders', count($orders), 'fas fa-shopping-cart'],
                            ['Total Revenue', $settings['currency'] . number_format($stats['total_revenue'] ?? 0, 2), 'fas fa-dollar-sign'],
                            ['Active Mods', count(array_filter($mods, function($m) { return $m['enabled'] ?? false; })) . '/' . count($builtinMods), 'fas fa-puzzle-piece'],
                            ['Custom Themes', max(0, count($themes) - count($defaultThemes)), 'fas fa-palette']
                        ];
                        
                        foreach ($statsData as $stat) {
                            echo '<div class="stat-card">';
                            echo '<div class="stat-value">' . $stat[1] . '</div>';
                            echo '<div class="stat-label">' . $stat[0] . '</div>';
                            echo '</div>';
                        }
                        echo '</div>';
                        
                        // Recent activity
                        echo '<h4 style="margin-top: 40px;"><i class="fas fa-clock"></i> Recent Activity</h4>';
                        $recentItems = array_slice(array_reverse($items), 0, 5);
                        if (!empty($recentItems)) {
                            echo '<div class="data-table-container">';
                            echo '<table class="data-table">';
                            echo '<thead><tr><th>Item</th><th>Type</th><th>Price</th><th>Views</th><th>Created</th></tr></thead>';
                            echo '<tbody>';
                            foreach ($recentItems as $item) {
                                echo '<tr>';
                                echo '<td><strong>' . sanitizeInput($item['title'], false) . '</strong></td>';
                                echo '<td><span class="item-type ' . $item['type'] . '">' . ucfirst(str_replace('_', ' ', $item['type'])) . '</span></td>';
                                echo '<td>' . formatPrice($item['price'], $settings['currency']) . '</td>';
                                echo '<td>' . ($item['views'] ?? 0) . '</td>';
                                echo '<td>' . date('M j, Y', $item['created']) . '</td>';
                                echo '</tr>';
                            }
                            echo '</tbody></table>';
                            echo '</div>';
                        } else {
                            echo '<p style="text-align: center; color: var(--text-secondary); padding: 40px;">No items yet. <a href="?admin=add_item" class="btn">Add your first item</a></p>';
                        }
                        
                        // System Information
                        echo '<h4 style="margin-top: 40px;"><i class="fas fa-info-circle"></i> System Information</h4>';
                        echo '<div class="card" style="padding: 20px;">';
                        echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">';
                        echo '<div><strong>Version:</strong><br>SmallMart ' . VERSION . '</div>';
                        echo '<div><strong>PHP Version:</strong><br>' . phpversion() . '</div>';
                        echo '<div><strong>Timezone:</strong><br>' . ($settings['timezone'] ?? 'America/New_York') . '</div>';
                        echo '<div><strong>PayPal Status:</strong><br>' . ((!empty($settings['mods']['paypal_integration']['settings']['paypal_email'])) ? 'Connected' : 'Not Configured') . '</div>';
                        echo '<div><strong>Available Mods:</strong><br>' . count($builtinMods) . ' built-in, ' . count(glob(MODS_DIR . '*.php')) . ' custom</div>';
                        echo '<div><strong>Backlink Status:</strong><br>' . (isDonationComplete() ? 'Removed' : 'Active') . '</div>';
                        echo '</div>';
                        echo '</div>';
                        break;
                        
                    case 'items':
                        
            requireAdmin();
echo '<h3><i class="fas fa-box"></i> Item Management</h3>';
                        echo '<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">';
                        echo '<p style="color: var(--text-secondary);">Manage all items in your marketplace</p>';
                        echo '<a href="?admin=add_item" class="btn"><i class="fas fa-plus"></i> Add New Item</a>';
                        echo '</div>';
                        
                        if (!empty($items)) {
                            echo '<div class="data-table-container">';
                            echo '<table class="data-table">';
                            echo '<thead><tr><th>Image</th><th>Title</th><th>Type</th><th>Price</th><th>Views</th><th>Payment Options</th><th>Actions</th></tr></thead>';
                            echo '<tbody>';
                            foreach (array_reverse($items) as $item) {
                                echo '<tr>';
                                echo '<td>';
                                if (!empty($item['images'])) {
                                    echo '<img src="' . UPLOADS_WEB_PATH . $item['images'][0] . '" style="width: 50px; height: 50px; object-fit: cover; border-radius: 6px;">';
                                } else {
                                    echo '<div style="width: 50px; height: 50px; background: var(--surface); border-radius: 6px; display: flex; align-items: center; justify-content: center;"><i class="fas fa-image"></i></div>';
                                }
                                echo '</td>';
                                echo '<td><strong>' . sanitizeInput($item['title'], false) . '</strong></td>';
                                echo '<td><span class="item-type ' . $item['type'] . '">' . ucfirst(str_replace('_', ' ', $item['type'])) . '</span></td>';
                                echo '<td>' . formatPrice($item['price'], $settings['currency']) . '</td>';
                                echo '<td>' . ($item['views'] ?? 0) . '</td>';
                                echo '<td>';
                                $paymentOptions = [];
                                if ($item['enable_paypal'] ?? true) $paymentOptions[] = '<i class="fab fa-paypal text-primary"></i>';
                                if ($item['enable_contact'] ?? true) $paymentOptions[] = '<i class="fas fa-envelope text-primary"></i>';
                                echo !empty($paymentOptions) ? implode(' ', $paymentOptions) : '<span style="color: var(--text-secondary);">None</span>';
                                echo '</td>';
                                echo '<td>';
                                echo '<div class="action-buttons">';
                                echo '<a href="?view=' . $item['id'] . '" class="btn btn-sm btn-info" title="View"><i class="fas fa-eye"></i> View</a>';
                                echo '<a href="?admin=edit_item&id=' . $item['id'] . '" class="btn btn-sm btn-warning" title="Edit"><i class="fas fa-edit"></i> Edit</a>';
                                echo '<a href="?action=delete_item&id=' . $item['id'] . '" class="btn btn-sm btn-danger" title="Delete" onclick="return confirm(\'Are you sure you want to delete this item?\')"><i class="fas fa-trash"></i> Delete</a>';
                                echo '</div>';
                                echo '</td>';
                                echo '</tr>';
                            }
                            echo '</tbody></table>';
                            echo '</div>';
                        } else {
                            echo '<div style="text-align: center; padding: 60px; background: var(--surface); border-radius: 12px;">';
                            echo '<i class="fas fa-box" style="font-size: 4rem; color: var(--text-secondary); margin-bottom: 20px;"></i>';
                            echo '<h4>No Items Yet</h4>';
                            echo '<p style="color: var(--text-secondary); margin: 15px 0;">Start building your marketplace by adding your first item.</p>';
                            echo '<a href="?admin=add_item" class="btn btn-lg"><i class="fas fa-plus"></i> Add Your First Item</a>';
                            echo '</div>';
                        }
                        break;
                        
                    case 'add_item':
                    
            requireAdmin();
case 'edit_item':
                        
            requireAdmin();
$isEdit = $adminPage === 'edit_item';
                        $editItem = null;
                        
                        if ($isEdit && isset($_GET['id'])) {
                            foreach ($items as $item) {
                                if ($item['id'] === $_GET['id']) {
                                    $editItem = $item;
                                    break;
                                }
                            }
                        }
                        
                        echo '<h3><i class="fas fa-' . ($isEdit ? 'edit' : 'plus') . '"></i> ' . ($isEdit ? 'Edit Item' : 'Add New Item') . '</h3>';
                        echo '<form method="post" action="?action=save_item" enctype="multipart/form-data">';
                        echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                        
                        if ($isEdit && $editItem) {
                            echo '<input type="hidden" name="id" value="' . $editItem['id'] . '">';
                            echo '<input type="hidden" name="created" value="' . ($editItem['created'] ?? time()) . '">';
                            echo '<input type="hidden" name="views" value="' . ($editItem['views'] ?? 0) . '">';
                        }
                        
                        echo '<div style="display: grid; grid-template-columns: 2fr 1fr; gap: 40px;">';
                        
                        // Main content
                        echo '<div>';
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Title *</label>';
                        echo '<input type="text" name="title" class="form-control" value="' . sanitizeInput($editItem['title'] ?? '', false) . '" required>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Description *</label>';
                        if ($settings['enable_wysiwyg'] ?? true) {
                            echo '<div id="description-editor" style="height: 300px;"></div>';
                            echo '<textarea name="description" id="description-textarea" style="display: none;" required>' . ($editItem['description'] ?? '') . '</textarea>';
                        } else {
                            echo '<textarea name="description" class="form-control" rows="8" required>' . sanitizeInput($editItem['description'] ?? '', true) . '</textarea>';
                        }
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Images (Max: ' . ($settings['max_images'] ?? MAX_IMAGES_PER_ITEM) . ')</label>';
                        echo '<input type="file" name="images[]" class="form-control" multiple accept="image/*">';
                        echo '<small style="color: var(--text-secondary); margin-top: 8px; display: block;">Supported formats: JPG, PNG, GIF, WebP. Max file size: 10MB each.</small>';
                        
                        if ($isEdit && !empty($editItem['images'])) {
                            echo '<div style="display: flex; gap: 10px; margin-top: 15px; flex-wrap: wrap;">';
                            foreach ($editItem['images'] as $image) {
                                echo '<div style="position: relative;">';
                                echo '<img src="' . UPLOADS_WEB_PATH . $image . '" style="width: 100px; height: 100px; object-fit: cover; border-radius: 8px;">';
                                echo '<input type="hidden" name="existing_images[]" value="' . $image . '">';
                                echo '</div>';
                            }
                            echo '</div>';
                        }
                        echo '</div>';
                        echo '</div>';
                        
                        // Sidebar
                        echo '<div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Type *</label>';
                        echo '<select name="type" class="form-control" required>';
                        if ($settings['enabled_modes']['buy_now']) {
                            $selected = ($editItem['type'] ?? '') === 'buy_now' ? 'selected' : '';
                            echo '<option value="buy_now" ' . $selected . '>Buy It Now</option>';
                        }
                        if ($settings['enabled_modes']['auction']) {
                            $selected = ($editItem['type'] ?? '') === 'auction' ? 'selected' : '';
                            echo '<option value="auction" ' . $selected . '>Auction</option>';
                        }
                        if ($settings['enabled_modes']['make_offer']) {
                            $selected = ($editItem['type'] ?? '') === 'make_offer' ? 'selected' : '';
                            echo '<option value="make_offer" ' . $selected . '>Make Offer</option>';
                        }
                        echo '</select>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Price (' . sanitizeInput($settings['currency'], false) . ') *</label>';
                        echo '<input type="text" name="price" class="form-control" value="' . sanitizeInput($editItem['price'] ?? '', false) . '" required placeholder="0.00 or text like \'Best Offer\'">';
                        echo '</div>';
                        
                        // Payment Options Section
                        echo '<div class="form-group">';
                        echo '<label class="form-label"><i class="fas fa-credit-card"></i> Payment Options for This Item</label>';
                        echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin-top: 5px;">';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; margin-bottom: 10px; cursor: pointer;">';
                        $paypalChecked = ($editItem['enable_paypal'] ?? true) ? 'checked' : '';
                        echo '<input type="checkbox" name="enable_paypal" ' . $paypalChecked . '>';
                        echo '<i class="fab fa-paypal" style="color: #0070ba;"></i> Enable PayPal for this item';
                        echo '</label>';
                        
                        echo '<label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">';
                        $contactChecked = ($editItem['enable_contact'] ?? true) ? 'checked' : '';
                        echo '<input type="checkbox" name="enable_contact" ' . $contactChecked . '>';
                        echo '<i class="fas fa-envelope" style="color: var(--primary);"></i> Enable "Contact Seller" for this item';
                        echo '</label>';
                        
                        echo '<small style="color: var(--text-secondary); display: block; margin-top: 10px;">These settings override global payment method settings for this specific item.</small>';
                        echo '</div>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Category</label>';
                        echo '<select name="category" class="form-control">';
                        echo '<option value="">Select Category</option>';
                        foreach ($categories as $catId => $category) {
                            if ($category['enabled'] ?? false) {
                                $selected = ($editItem['category'] ?? '') === $catId ? 'selected' : '';
                                echo '<option value="' . $catId . '" ' . $selected . '>' . sanitizeInput($category['name'], false) . '</option>';
                            }
                        }
                        echo '</select>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Condition</label>';
                        echo '<select name="condition" class="form-control">';
                        $conditions = ['new' => 'New', 'used' => 'Used', 'refurbished' => 'Refurbished', 'for_parts' => 'For Parts'];
                        foreach ($conditions as $value => $label) {
                            $selected = ($editItem['condition'] ?? 'new') === $value ? 'selected' : '';
                            echo '<option value="' . $value . '" ' . $selected . '>' . $label . '</option>';
                        }
                        echo '</select>';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Stock Quantity</label>';
                        echo '<input type="number" name="stock" class="form-control" value="' . ($editItem['stock'] ?? 1) . '" min="0">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Location</label>';
                        echo '<input type="text" name="location" class="form-control" value="' . sanitizeInput($editItem['location'] ?? '', false) . '" placeholder="City, State">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Shipping Cost</label>';
                        echo '<input type="text" name="shipping_cost" class="form-control" value="' . ($editItem['shipping_cost'] ?? '0') . '" placeholder="0.00">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label class="form-label">Tags</label>';
                        echo '<input type="text" name="tags" class="form-control" value="' . sanitizeInput($editItem['tags'] ?? '', false) . '" placeholder="tag1, tag2, tag3">';
                        echo '</div>';
                        
                        echo '<div class="form-group">';
                        echo '<label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">';
                        $featured = ($editItem['featured'] ?? false) ? 'checked' : '';
                        echo '<input type="checkbox" name="featured" ' . $featured . '>';
                        echo '<i class="fas fa-star" style="color: var(--primary);"></i> Featured Item';
                        echo '</label>';
                        echo '</div>';
                        
                        echo '<button type="submit" class="btn btn-lg" style="width: 100%; margin-top: 30px;">';
                        echo '<i class="fas fa-save"></i> ' . ($isEdit ? 'Update Item' : 'Create Item');
                        echo '</button>';
                        
                        echo '</div>';
                        echo '</div>';
                        echo '</form>';
                        break;
                        
                    case 'orders':
                        echo '<h3><i class="fas fa-shopping-cart"></i> Order Management</h3>';
                        echo '<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">';
                        echo '<p style="color: var(--text-secondary);">Manage all customer orders and transactions</p>';
                        if (!empty($orders)) {
                            echo '<a href="?action=clear_orders" class="btn btn-danger" onclick="return confirm(\'Are you sure you want to clear ALL order history? This cannot be undone!\')"><i class="fas fa-trash-alt"></i> Clear Order History</a>';
                        }
                        echo '</div>';
                        
                        if (!empty($orders)) {
                            echo '<div class="data-table-container">';
                            echo '<table class="data-table">';
                            echo '<thead><tr><th>Order ID</th><th>Item</th><th>Status</th><th>Payment Method</th><th>Date</th></tr></thead>';
                            echo '<tbody>';
                            foreach (array_reverse($orders) as $order) {
                                $orderItem = null;
                                foreach ($items as $item) {
                                    if ($item['id'] === $order['item_id']) {
                                        $orderItem = $item;
                                        break;
                                    }
                                }
                                
                                echo '<tr>';
                                echo '<td><strong>' . substr($order['id'], -8) . '</strong></td>';
                                echo '<td>' . ($orderItem ? sanitizeInput($orderItem['title'], false) : 'Item not found') . '</td>';
                                echo '<td><span class="item-type ' . ($order['status'] === 'completed' ? 'buy_now' : 'auction') . '">' . ucfirst($order['status']) . '</span></td>';
                                echo '<td><i class="fab fa-paypal"></i> ' . ucfirst($order['payment_method']) . '</td>';
                                echo '<td>' . date('M j, Y H:i', strtotime($order['created'])) . '</td>';
                                echo '</tr>';
                            }
                            echo '</tbody></table>';
                            echo '</div>';
                        } else {
                            echo '<div style="text-align: center; padding: 60px;">';
                            echo '<i class="fas fa-shopping-cart" style="font-size: 4rem; color: var(--text-secondary); margin-bottom: 20px;"></i>';
                            echo '<h4>No Orders Yet</h4>';
                            echo '<p style="color: var(--text-secondary);">Orders will appear here when customers make purchases.</p>';
                            echo '</div>';
                        }
                        break;
                        
                    case 'mods':
    $modTab = safeGet($_GET, 'tab', 'builtin');
    $editMod = safeGet($_GET, 'edit');
    
    if ($editMod && isset($mods[$editMod])) {
        $mod = $mods[$editMod];
        echo '<h3><i class="fas fa-cog"></i> Configure ' . sanitizeInput($mod['name'], false) . '</h3>';
        echo '<p style="color: var(--text-secondary); margin-bottom: 30px;">' . sanitizeInput($mod['description'], false) . '</p>';
        
        echo '<form method="post" action="?action=save_mod_settings">';
        echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
        echo '<input type="hidden" name="mod_id" value="' . $editMod . '">';
        
        foreach ($mod['settings'] ?? [] as $key => $value) {
            echo '<div class="form-group">';
            echo '<label class="form-label">' . ucfirst(str_replace('_', ' ', $key)) . '</label>';
            
            if (is_bool($value)) {
                echo '<label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">';
                echo '<input type="checkbox" name="mod_setting_' . $key . '" ' . ($value ? 'checked' : '') . '>';
                echo 'Enable this feature';
                echo '</label>';
            } elseif (is_numeric($value)) {
                echo '<input type="number" name="mod_setting_' . $key . '" class="form-control" value="' . $value . '">';
            } elseif (is_array($value)) {
                echo '<textarea name="mod_setting_' . $key . '" class="form-control" rows="3">' . implode(', ', $value) . '</textarea>';
            } else {
                echo '<input type="text" name="mod_setting_' . $key . '" class="form-control" value="' . sanitizeInput($value, false) . '">';
            }
            echo '</div>';
        }
        
        echo '<div style="display: flex; gap: 15px; margin-top: 30px;">';
        echo '<button type="submit" class="btn"><i class="fas fa-save"></i> Save Settings</button>';
        echo '<a href="?admin=mods" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to Mods</a>';
        echo '</div>';
        echo '</form>';
        
    } else {
        echo '<h3><i class="fas fa-puzzle-piece"></i> Enhanced Mod System Management</h3>';
        echo '<p style="color: var(--text-secondary); margin-bottom: 30px;">Manage built-in system mods and upload custom mods to extend functionality.</p>';
        
        // Tab Navigation
        echo '<div style="display: flex; gap: 10px; margin-bottom: 30px; border-bottom: 1px solid var(--border); padding-bottom: 15px;">';
        echo '<a href="?admin=mods&tab=builtin" class="btn ' . ($modTab === 'builtin' ? '' : 'btn-secondary') . '">Built-in Mods</a>';
        echo '<a href="?admin=mods&tab=custom" class="btn ' . ($modTab === 'custom' ? '' : 'btn-secondary') . '">Upload Custom</a>';
        echo '<a href="?admin=mods&tab=documentation" class="btn ' . ($modTab === 'documentation' ? '' : 'btn-secondary') . '">Documentation</a>';
        echo '</div>';
        
        switch ($modTab) {
            case 'custom':
                echo '<h4><i class="fas fa-upload"></i> Upload Custom Mod</h4>';
                echo '<div class="mod-upload-section">';
                echo '<form method="post" action="?action=upload_mod" enctype="multipart/form-data">';
                echo '<input type="hidden" name="csrf_token" value="' . generateCSRF() . '">';
                echo '<div class="form-group">';
                echo '<label class="form-label">Mod File (PHP)</label>';
                echo '<input type="file" name="mod_file" class="form-control" accept=".php" required>';
                echo '</div>';
                echo '<div class="form-group">';
                echo '<label class="form-label">Mod ID</label>';
                echo '<input type="text" name="mod_id" class="form-control" placeholder="my_custom_mod" required>';
                echo '</div>';
                echo '<button type="submit" class="btn"><i class="fas fa-upload"></i> Upload Mod</button>';
                echo '</form>';
                echo '</div>';
                
                $customModFiles = glob(MODS_DIR . '*.php');
                if (!empty($customModFiles)) {
                    echo '<h4><i class="fas fa-list"></i> Installed Custom Mods</h4>';
                    echo '<div class="data-table-container">';
                    echo '<table class="data-table">';
                    echo '<thead><tr><th>Mod Name</th><th>File Size</th><th>Actions</th></tr></thead>';
                    echo '<tbody>';
                    foreach ($customModFiles as $modFile) {
                        $modId = basename($modFile, '.php');
                        $fileSize = formatBytes(filesize($modFile));
                        echo '<tr>';
                        echo '<td><strong>' . $modId . '</strong></td>';
                        echo '<td>' . $fileSize . '</td>';
                        echo '<td>';
                        echo '<a href="?action=delete_mod&id=' . $modId . '" class="btn btn-sm btn-danger" onclick="return confirm(\'Delete this mod?\')"><i class="fas fa-trash"></i> Delete</a>';
                        echo '</td>';
                        echo '</tr>';
                    }
                    echo '</tbody></table>';
                    echo '</div>';
                }
                break;
                
            case 'documentation':
                echo '<div class="mod-docs">';
                
                // Header
                echo '<h4><i class="fas fa-book"></i> SmallMart Mod Development Guide v6.1.5</h4>';
                echo '<p style="font-size: 1.1rem; color: var(--text-secondary); margin-bottom: 30px;">Complete API reference and development guide for creating powerful custom modifications for SmallMart. Build professional extensions using our hook-based architecture.</p>';
                
                // Table of Contents
                echo '<div style="background: var(--surface); padding: 25px; border-radius: 12px; margin-bottom: 40px; border: 2px solid var(--border);">';
                echo '<h5 style="margin-top: 0; color: var(--primary);"><i class="fas fa-list"></i> Documentation Contents</h5>';
                echo '<div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px;">';
                echo '<a href="#quick-start" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none; transition: all 0.3s;">→ Quick Start Guide</a>';
                echo '<a href="#mod-structure" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Mod File Structure</a>';
                echo '<a href="#api-reference" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Complete API Reference</a>';
                echo '<a href="#hooks" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Available Hooks</a>';
                echo '<a href="#examples" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Advanced Examples</a>';
                echo '<a href="#security" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Security Guidelines</a>';
                echo '<a href="#best-practices" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Best Practices</a>';
                echo '<a href="#troubleshooting" style="color: var(--text); padding: 10px; background: var(--card); border-radius: 8px; text-decoration: none;">→ Troubleshooting</a>';
                echo '</div>';
                echo '</div>';
                
                // Quick Start
                echo '<div id="quick-start" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-rocket"></i> Quick Start Guide</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">Create your first SmallMart mod in minutes. Mods are PHP files placed in the <code style="background: var(--surface); padding: 3px 8px; border-radius: 4px; color: var(--primary);">/mods/</code> directory that extend core functionality without modifying the main codebase.</p>';
                
                echo '<div style="background: linear-gradient(135deg, var(--primary), var(--secondary)); padding: 25px; border-radius: 12px; margin: 20px 0;">';
                echo '<h6 style="margin-top: 0; color: white; font-size: 1.1rem;">Basic Mod Template</h6>';
                echo '<p style="color: rgba(255,255,255,0.9); margin-bottom: 15px;">Copy this template to create your first mod:</p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 25px; border-radius: 8px; overflow-x: auto; margin: 0; border: 2px solid rgba(255,255,255,0.1);">';
                echo htmlspecialchars('<?php
/*
SmallMart Mod: My Custom Mod
Description: Adds custom functionality to SmallMart
Version: 1.0.0
Author: Your Name
Author URI: https://yourwebsite.com
License: MIT
*/

// Prevent direct access
if (!defined(\'VERSION\')) {
    exit(\'Direct access not permitted\');
}

/**
 * Initialize the mod
 * This function runs when the mod is loaded
 */
function my_custom_mod_init() {
    // Register hooks
    SmallMartModAPI::addHook(\'item_view\', \'my_custom_mod_handler\', 10);
    
    // Set default settings
    SmallMartModAPI::setSetting(\'my_mod_enabled\', true);
    SmallMartModAPI::setSetting(\'my_mod_color\', \'#ff0000\');
    
    // Log activation
    SmallMartModAPI::log(\'My Custom Mod activated successfully\', \'info\');
}

/**
 * Process item view hook
 * @param array $data Item data array
 * @return array Modified item data
 */
function my_custom_mod_handler($data) {
    if (SmallMartModAPI::getSetting(\'my_mod_enabled\', false)) {
        // Your custom logic here
        $data[\'custom_field\'] = \'Custom Value\';
        SmallMartModAPI::log(\'Item modified by custom mod\', \'info\');
    }
    return $data;
}

// Auto-initialize when mod is loaded
my_custom_mod_init();
');
                echo '</pre>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 20px; border-radius: 8px; border-left: 4px solid var(--accent);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-info-circle"></i> Installation Steps</h6>';
                echo '<ol style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li>Create a new file: <code style="background: var(--surface); padding: 2px 6px; border-radius: 4px;">/mods/my_custom_mod.php</code></li>';
                echo '<li>Copy the template code above into your file</li>';
                echo '<li>Customize the mod header with your information</li>';
                echo '<li>Add your custom logic to the handler function</li>';
                echo '<li>Save the file - it will be loaded automatically</li>';
                echo '<li>Check the Console → Mod Log to verify activation</li>';
                echo '</ol>';
                echo '</div>';
                echo '</div>';
                
                // Mod Structure
                echo '<div id="mod-structure" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-sitemap"></i> Mod File Structure</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">Understanding proper structure ensures seamless integration with SmallMart.</p>';
                
                echo '<h6 style="color: var(--primary); margin-top: 25px;">Required Header Block</h6>';
                echo '<p>Every mod MUST include a header comment with metadata:</p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 20px; border-radius: 8px;">';
                echo htmlspecialchars('/*
SmallMart Mod: [Mod Name] (REQUIRED)
Description: [Brief description] (REQUIRED)
Version: [1.0.0] (REQUIRED)
Author: [Your Name] (Optional)
Author URI: [https://yoursite.com] (Optional)
License: [MIT/GPL/etc] (Optional)
Requires: [5.0+] (Optional - Minimum SmallMart version)
*/');
                echo '</pre>';
                
                echo '<h6 style="color: var(--primary); margin-top: 25px;">File Organization</h6>';
                echo '<div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin-top: 15px;">';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">1. Header Comment</strong><br>';
                echo '<small style="color: var(--text-secondary);">Mod metadata and information</small>';
                echo '</div>';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">2. Security Check</strong><br>';
                echo '<small style="color: var(--text-secondary);">Prevent direct file access</small>';
                echo '</div>';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">3. Init Function</strong><br>';
                echo '<small style="color: var(--text-secondary);">Register hooks and settings</small>';
                echo '</div>';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">4. Handler Functions</strong><br>';
                echo '<small style="color: var(--text-secondary);">Your custom logic</small>';
                echo '</div>';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">5. Helper Functions</strong><br>';
                echo '<small style="color: var(--text-secondary);">Utility functions</small>';
                echo '</div>';
                echo '<div style="background: var(--card); padding: 15px; border-radius: 8px;">';
                echo '<strong style="color: var(--accent);">6. Auto-Initialize</strong><br>';
                echo '<small style="color: var(--text-secondary);">Call init function at end</small>';
                echo '</div>';
                echo '</div>';
                echo '</div>';
                
                // API Reference
                echo '<div id="api-reference" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-code"></i> Complete API Reference</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">The SmallMartModAPI class provides all methods needed for professional mod development.</p>';
                
                // addHook
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border: 2px solid var(--border);">';
                echo '<h6 style="margin-top: 0;"><code style="color: var(--primary); font-size: 1.1rem;">SmallMartModAPI::addHook($hookName, $callback, $priority = 10)</code></h6>';
                echo '<p><strong style="color: var(--accent);">Description:</strong> Register a callback function to execute when a specific hook is triggered. Multiple callbacks can be registered to the same hook with different priorities.</p>';
                echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin: 15px 0;">';
                echo '<strong>Parameters:</strong>';
                echo '<ul style="margin: 10px 0; padding-left: 20px;">';
                echo '<li><code style="color: var(--primary);">$hookName</code> <em>(string)</em> - Name of the hook to attach to</li>';
                echo '<li><code style="color: var(--primary);">$callback</code> <em>(callable)</em> - Function or method name to execute</li>';
                echo '<li><code style="color: var(--primary);">$priority</code> <em>(int)</em> - Execution order (lower = earlier, default: 10)</li>';
                echo '</ul>';
                echo '</div>';
                echo '<p><strong style="color: var(--accent);">Returns:</strong> <code>void</code></p>';
                echo '<p><strong style="color: var(--accent);">Example:</strong></p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 15px; border-radius: 8px; margin: 0;">SmallMartModAPI::addHook(\'item_view\', \'my_function\', 5);
SmallMartModAPI::addHook(\'search_results\', \'filter_results\', 20);</pre>';
                echo '</div>';
                
                // executeHook
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border: 2px solid var(--border);">';
                echo '<h6 style="margin-top: 0;"><code style="color: var(--primary); font-size: 1.1rem;">SmallMartModAPI::executeHook($hookName, $data = null)</code></h6>';
                echo '<p><strong style="color: var(--accent);">Description:</strong> Execute all registered callbacks for a hook in priority order, passing data through the chain. Each callback receives the modified data from the previous callback.</p>';
                echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin: 15px 0;">';
                echo '<strong>Parameters:</strong>';
                echo '<ul style="margin: 10px 0; padding-left: 20px;">';
                echo '<li><code style="color: var(--primary);">$hookName</code> <em>(string)</em> - Hook name to execute</li>';
                echo '<li><code style="color: var(--primary);">$data</code> <em>(mixed)</em> - Data to pass through callbacks</li>';
                echo '</ul>';
                echo '</div>';
                echo '<p><strong style="color: var(--accent);">Returns:</strong> <code>mixed</code> - Modified data after all callbacks</p>';
                echo '<p><strong style="color: var(--accent);">Example:</strong></p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 15px; border-radius: 8px; margin: 0;">$itemData = SmallMartModAPI::executeHook(\'item_view\', $itemData);
$searchResults = SmallMartModAPI::executeHook(\'search_results\', $results);</pre>';
                echo '</div>';
                
                // getSetting
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border: 2px solid var(--border);">';
                echo '<h6 style="margin-top: 0;"><code style="color: var(--primary); font-size: 1.1rem;">SmallMartModAPI::getSetting($key, $default = null)</code></h6>';
                echo '<p><strong style="color: var(--accent);">Description:</strong> Retrieve a mod setting value with optional default fallback. Settings persist across requests.</p>';
                echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin: 15px 0;">';
                echo '<strong>Parameters:</strong>';
                echo '<ul style="margin: 10px 0; padding-left: 20px;">';
                echo '<li><code style="color: var(--primary);">$key</code> <em>(string)</em> - Setting key to retrieve</li>';
                echo '<li><code style="color: var(--primary);">$default</code> <em>(mixed)</em> - Default value if setting doesn\'t exist</li>';
                echo '</ul>';
                echo '</div>';
                echo '<p><strong style="color: var(--accent);">Returns:</strong> <code>mixed</code> - Setting value or default</p>';
                echo '<p><strong style="color: var(--accent);">Example:</strong></p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 15px; border-radius: 8px; margin: 0;">$enabled = SmallMartModAPI::getSetting(\'my_mod_enabled\', false);
$color = SmallMartModAPI::getSetting(\'theme_color\', \'#ff0000\');
$limit = SmallMartModAPI::getSetting(\'item_limit\', 10);</pre>';
                echo '</div>';
                
                // setSetting
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border: 2px solid var(--border);">';
                echo '<h6 style="margin-top: 0;"><code style="color: var(--primary); font-size: 1.1rem;">SmallMartModAPI::setSetting($key, $value)</code></h6>';
                echo '<p><strong style="color: var(--accent);">Description:</strong> Store a mod setting value for persistent configuration across sessions. Supports strings, integers, booleans, and arrays.</p>';
                echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin: 15px 0;">';
                echo '<strong>Parameters:</strong>';
                echo '<ul style="margin: 10px 0; padding-left: 20px;">';
                echo '<li><code style="color: var(--primary);">$key</code> <em>(string)</em> - Setting key to store</li>';
                echo '<li><code style="color: var(--primary);">$value</code> <em>(mixed)</em> - Value to store (string, int, bool, array)</li>';
                echo '</ul>';
                echo '</div>';
                echo '<p><strong style="color: var(--accent);">Returns:</strong> <code>void</code></p>';
                echo '<p><strong style="color: var(--accent);">Example:</strong></p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 15px; border-radius: 8px; margin: 0;">SmallMartModAPI::setSetting(\'my_mod_enabled\', true);
SmallMartModAPI::setSetting(\'max_items\', 50);
SmallMartModAPI::setSetting(\'allowed_types\', [\'book\', \'video\']);</pre>';
                echo '</div>';
                
                // log
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border: 2px solid var(--border);">';
                echo '<h6 style="margin-top: 0;"><code style="color: var(--primary); font-size: 1.1rem;">SmallMartModAPI::log($message, $level = \'info\')</code></h6>';
                echo '<p><strong style="color: var(--accent);">Description:</strong> Write entries to the mod log file for debugging and monitoring. View logs in Console → Mod Log.</p>';
                echo '<div style="background: var(--surface); padding: 15px; border-radius: 8px; margin: 15px 0;">';
                echo '<strong>Parameters:</strong>';
                echo '<ul style="margin: 10px 0; padding-left: 20px;">';
                echo '<li><code style="color: var(--primary);">$message</code> <em>(string)</em> - Log message text</li>';
                echo '<li><code style="color: var(--primary);">$level</code> <em>(string)</em> - Log level: info, warning, error, debug</li>';
                echo '</ul>';
                echo '</div>';
                echo '<p><strong style="color: var(--accent);">Returns:</strong> <code>void</code></p>';
                echo '<p><strong style="color: var(--accent);">Example:</strong></p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 15px; border-radius: 8px; margin: 0;">SmallMartModAPI::log(\'Mod initialized successfully\', \'info\');
SmallMartModAPI::log(\'Missing configuration value\', \'warning\');
SmallMartModAPI::log(\'Critical error in processing\', \'error\');
SmallMartModAPI::log(\'Processing item ID: 12345\', \'debug\');</pre>';
                echo '</div>';
                echo '</div>';
                
                // Available Hooks
                echo '<div id="hooks" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-plug"></i> Available Hooks</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">SmallMart provides 40+ hooks throughout the system. Each hook receives specific data and expects specific return values.</p>';
                
                echo '<div class="data-table-container">';
                echo '<table class="data-table">';
                echo '<thead><tr><th>Hook Name</th><th>Trigger Point</th><th>Data Passed</th><th>Expected Return</th></tr></thead>';
                echo '<tbody>';
                
                $hooks = [
                    ['item_view', 'When viewing single item', 'array $itemData', 'Modified $itemData'],
                    ['item_list', 'Item grid display', 'array $items', 'Modified $items'],
                    ['search_form', 'Search form render', 'array $formData', 'Modified $formData'],
                    ['search_results', 'After search executes', 'array $results', 'Modified $results'],
                    ['search_before', 'Before search query', 'string $query', 'Modified $query'],
                    ['search_after', 'After search completes', 'array $results', 'Modified $results'],
                    ['item_filters', 'During filtering', 'array $filters', 'Modified $filters'],
                    ['payment_methods', 'Payment options page', 'array $methods', 'Modified $methods'],
                    ['payment_form', 'Payment form generation', 'array $formData', 'Modified $formData'],
                    ['payment_validation', 'Payment validation', 'array $paymentData', 'bool success'],
                    ['order_processing', 'Order creation', 'array $orderData', 'Modified $orderData'],
                    ['admin_menu', 'Admin menu generation', 'array $menuItems', 'Modified $menuItems'],
                    ['theme_settings', 'Theme config page', 'array $settings', 'Modified $settings'],
                    ['theme_export', 'Theme export process', 'array $themeData', 'Modified $themeData'],
                    ['theme_import', 'Theme import process', 'array $themeData', 'Modified $themeData'],
                    ['page_view', 'Every page load', 'array $pageData', 'Modified $pageData'],
                    ['analytics_dashboard', 'Analytics display', 'array $stats', 'Modified $stats'],
                    ['tracking_script', 'Tracking code output', 'string $script', 'Modified $script'],
                    ['analytics_report', 'Report generation', 'array $reportData', 'Modified $reportData'],
                    ['head_meta', 'HTML head section', 'array $metaTags', 'Modified $metaTags'],
                    ['sitemap_generation', 'Sitemap creation', 'array $urls', 'Modified $urls'],
                    ['schema_markup', 'Schema.org output', 'array $schema', 'Modified $schema'],
                    ['seo_analysis', 'SEO analysis', 'array $seoData', 'Modified $seoData'],
                    ['image_upload', 'Image upload process', 'array $imageData', 'Modified $imageData'],
                    ['image_display', 'Image rendering', 'string $imageHTML', 'Modified $imageHTML'],
                    ['lazy_loading', 'Image lazy load', 'array $imageAttr', 'Modified $imageAttr'],
                    ['order_placed', 'New order created', 'array $orderData', 'void'],
                    ['inquiry_received', 'Contact form submit', 'array $inquiryData', 'void'],
                    ['admin_alerts', 'Admin notification', 'array $alertData', 'void'],
                    ['stock_update', 'Inventory change', 'array $stockData', 'Modified $stockData'],
                    ['low_stock_alert', 'Low stock trigger', 'array $itemData', 'void'],
                    ['inventory_report', 'Inventory report', 'array $reportData', 'Modified $reportData'],
                    ['checkout_process', 'Checkout flow', 'array $checkoutData', 'Modified $checkoutData'],
                    ['coupon_validation', 'Coupon check', 'array $couponData', 'bool valid'],
                    ['discount_calculation', 'Discount compute', 'array $discountData', 'float discount'],
                    ['comparison_widget', 'Comparison UI', 'array $compareData', 'Modified $compareData'],
                    ['product_features', 'Feature list', 'array $features', 'Modified $features'],
                    ['comparison_table', 'Comparison table', 'array $tableData', 'Modified $tableData'],
                    ['request_end', 'Request completion', 'null', 'void']
                ];
                
                foreach ($hooks as $hook) {
                    echo '<tr>';
                    echo '<td><code style="color: var(--primary); font-weight: bold; font-size: 0.95rem;">' . $hook[0] . '</code></td>';
                    echo '<td>' . $hook[1] . '</td>';
                    echo '<td><code style="font-size: 0.9rem;">' . $hook[2] . '</code></td>';
                    echo '<td><code style="font-size: 0.9rem;">' . $hook[3] . '</code></td>';
                    echo '</tr>';
                }
                
                echo '</tbody></table>';
                echo '</div>';
                echo '</div>';
                
                // Advanced Examples
                echo '<div id="examples" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-lightbulb"></i> Advanced Examples</h5>';
                
                echo '<h6 style="color: var(--primary); margin-top: 30px;">Example 1: Add Custom Item Fields</h6>';
                echo '<p>Add warranty information and premium badges to items:</p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 25px; border-radius: 8px; overflow-x: auto;">';
                echo htmlspecialchars('<?php
/*
SmallMart Mod: Custom Item Fields
Description: Adds warranty info and premium badges
Version: 1.0.0
Author: SmallMart
*/

if (!defined(\'VERSION\')) exit;

function custom_fields_init() {
    SmallMartModAPI::addHook(\'item_view\', \'add_custom_fields\');
    SmallMartModAPI::addHook(\'item_list\', \'add_custom_fields_bulk\');
    SmallMartModAPI::log(\'Custom Fields Mod loaded\');
}

function add_custom_fields($item) {
    // Add warranty information
    $item[\'warranty_years\'] = 2;
    $item[\'warranty_type\'] = \'manufacturer\';
    $item[\'warranty_contact\'] = \'1-800-WARRANTY\';
    
    // Add premium badge for expensive items
    if (isset($item[\'price\']) && is_numeric($item[\'price\']) && $item[\'price\'] > 1000) {
        $item[\'is_premium\'] = true;
        $item[\'premium_badge\'] = \'⭐ Premium Item\';
    }
    
    // Add trust indicators
    $item[\'trust_score\'] = 95;
    $item[\'verified_seller\'] = true;
    
    SmallMartModAPI::log(\'Custom fields added to item: \' . $item[\'id\']);
    
    return $item;
}

function add_custom_fields_bulk($items) {
    return array_map(\'add_custom_fields\', $items);
}

custom_fields_init();
');
                echo '</pre>';
                
                echo '<h6 style="color: var(--primary); margin-top: 40px;">Example 2: Custom Payment Gateway (Stripe)</h6>';
                echo '<p>Integrate Stripe payment processing:</p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 25px; border-radius: 8px; overflow-x: auto;">';
                echo htmlspecialchars('<?php
/*
SmallMart Mod: Stripe Payment Gateway
Description:  Stripe integration
Version: 1.0.0
Author: SmallMart
*/

if (!defined(\'VERSION\')) exit;

function stripe_payment_init() {
    // Register hooks
    SmallMartModAPI::addHook(\'payment_methods\', \'add_stripe_method\');
    SmallMartModAPI::addHook(\'payment_form\', \'stripe_payment_form\');
    SmallMartModAPI::addHook(\'payment_validation\', \'validate_stripe_payment\');
    
    // Set default configuration
    SmallMartModAPI::setSetting(\'stripe_public_key\', \'pk_test_...\');
    SmallMartModAPI::setSetting(\'stripe_secret_key\', \'sk_test_...\');
    SmallMartModAPI::setSetting(\'stripe_currency\', \'usd\');
    
    SmallMartModAPI::log(\'Stripe Payment Gateway initialized\');
}

function add_stripe_method($methods) {
    $methods[\'stripe\'] = [
        \'name\' => \'Credit Card (Stripe)\',
        \'enabled\' => true,
        \'icon\' => \'fas fa-credit-card\',
        \'description\' => \'Secure payment with Stripe\'
    ];
    return $methods;
}

function stripe_payment_form($formData) {
    if ($formData[\'method\'] === \'stripe\') {
        $publicKey = SmallMartModAPI::getSetting(\'stripe_public_key\');
        
        $formData[\'html\'] = \'
            <div style="max-width: 400px; margin: 20px 0;">
                <div id="stripe-card-element" style="padding: 15px; border: 2px solid #e0e0e0; border-radius: 8px;"></div>
                <div id="stripe-card-errors" style="color: #dc2626; margin-top: 10px;"></div>
            </div>
            <script src="https://js.stripe.com/v3/"></script>
            <script>
                var stripe = Stripe("\' . $publicKey . \'");
                var elements = stripe.elements();
                var cardElement = elements.create("card", {
                    style: {
                        base: {
                            fontSize: "16px",
                            color: "#32325d",
                            "::placeholder": { color: "#aab7c4" }
                        }
                    }
                });
                cardElement.mount("#stripe-card-element");
                
                cardElement.on("change", function(event) {
                    var displayError = document.getElementById("stripe-card-errors");
                    displayError.textContent = event.error ? event.error.message : "";
                });
            </script>
        \';
    }
    return $formData;
}

function validate_stripe_payment($paymentData) {
    if ($paymentData[\'method\'] === \'stripe\') {
        $secretKey = SmallMartModAPI::getSetting(\'stripe_secret_key\');
        
        try {
            // Initialize Stripe API
            // Create payment intent
            // Process payment
            
            SmallMartModAPI::log(\'Stripe payment processed successfully\', \'info\');
            return true;
            
        } catch (Exception $e) {
            SmallMartModAPI::log(\'Stripe payment failed: \' . $e->getMessage(), \'error\');
            return false;
        }
    }
    return $paymentData;
}

stripe_payment_init();
');
                echo '</pre>';
                
                echo '<h6 style="color: var(--primary); margin-top: 40px;">Example 3: Advanced Analytics Tracking</h6>';
                echo '<p>Integrate Google Analytics 4 with detailed event tracking:</p>';
                echo '<pre style="background: #1e1e1e; color: #f8f8f2; padding: 25px; border-radius: 8px; overflow-x: auto;">';
                echo htmlspecialchars('<?php
/*
SmallMart Mod: Google Analytics 4
Description: Complete GA4 integration with events
Version: 1.0.0
Author: SmallMart
*/

if (!defined(\'VERSION\')) exit;

function ga4_tracking_init() {
    SmallMartModAPI::addHook(\'head_meta\', \'ga4_tracking_script\');
    SmallMartModAPI::addHook(\'item_view\', \'ga4_track_item_view\');
    SmallMartModAPI::addHook(\'search_after\', \'ga4_track_search\');
    SmallMartModAPI::addHook(\'order_placed\', \'ga4_track_purchase\');
    
    SmallMartModAPI::setSetting(\'ga4_measurement_id\', \'G-XXXXXXXXXX\');
    SmallMartModAPI::setSetting(\'ga4_enabled\', true);
}

function ga4_tracking_script($meta) {
    if (!SmallMartModAPI::getSetting(\'ga4_enabled\', false)) {
        return $meta;
    }
    
    $measurementId = SmallMartModAPI::getSetting(\'ga4_measurement_id\');
    
    $meta[\'ga4_script\'] = "
        <script async src=\\"https://www.googletagmanager.com/gtag/js?id={$measurementId}\\"></script>
        <script>
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            gtag(\'js\', new Date());
            gtag(\'config\', \'{$measurementId}\', {
                \'send_page_view\': true,
                \'cookie_flags\': \'SameSite=None;Secure\'
            });
        </script>
    ";
    
    return $meta;
}

function ga4_track_item_view($item) {
    if (SmallMartModAPI::getSetting(\'ga4_enabled\', false)) {
        $trackingCode = "
            <script>
                gtag(\'event\', \'view_item\', {
                    \'currency\': \'USD\',
                    \'value\': " . ($item[\'price\'] ?? 0) . ",
                    \'items\': [{
                        \'item_id\': \'" . ($item[\'id\'] ?? \'\') . "\',
                        \'item_name\': \'" . addslashes($item[\'title\'] ?? \'\') . "\',
                        \'item_category\': \'" . ($item[\'category\'] ?? \'\') . "\',
                        \'price\': " . ($item[\'price\'] ?? 0) . "
                    }]
                });
            </script>
        ";
        
        $item[\'_ga4_tracking\'] = $trackingCode;
        SmallMartModAPI::log(\'GA4 item view tracked: \' . $item[\'id\']);
    }
    
    return $item;
}

function ga4_track_search($results) {
    if (SmallMartModAPI::getSetting(\'ga4_enabled\', false)) {
        $searchTerm = $_GET[\'q\'] ?? \'\';
        if ($searchTerm) {
            SmallMartModAPI::log(\'GA4 search tracked: \' . $searchTerm);
        }
    }
    return $results;
}

function ga4_track_purchase($orderData) {
    if (SmallMartModAPI::getSetting(\'ga4_enabled\', false)) {
        SmallMartModAPI::log(\'GA4 purchase tracked: \' . $orderData[\'id\']);
    }
}

ga4_tracking_init();
');
                echo '</pre>';
                echo '</div>';
                
                // Security Guidelines
                echo '<div id="security" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-shield-alt"></i> Security Guidelines</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">Follow these security best practices to protect your mod and SmallMart installation:</p>';
                
                echo '<div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 20px; margin-top: 25px;">';
                
                echo '<div style="background: var(--card); padding: 20px; border-radius: 8px; border-left: 4px solid #dc2626;">';
                echo '<h6 style="margin-top: 0; color: #dc2626;"><i class="fas fa-exclamation-triangle"></i> Input Validation</h6>';
                echo '<ul style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li>Always sanitize user input</li>';
                echo '<li>Validate data types and formats</li>';
                echo '<li>Use whitelist validation</li>';
                echo '<li>Never trust $_GET, $_POST, $_COOKIE</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 20px; border-radius: 8px; border-left: 4px solid #dc2626;">';
                echo '<h6 style="margin-top: 0; color: #dc2626;"><i class="fas fa-database"></i> Data Protection</h6>';
                echo '<ul style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li>Use prepared statements</li>';
                echo '<li>Never store passwords in plaintext</li>';
                echo '<li>Encrypt sensitive data</li>';
                echo '<li>Validate file uploads</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 20px; border-radius: 8px; border-left: 4px solid #dc2626;">';
                echo '<h6 style="margin-top: 0; color: #dc2626;"><i class="fas fa-lock"></i> Access Control</h6>';
                echo '<ul style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li>Prevent direct file access</li>';
                echo '<li>Check user permissions</li>';
                echo '<li>Use CSRF tokens</li>';
                echo '<li>Implement rate limiting</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 20px; border-radius: 8px; border-left: 4px solid #dc2626;">';
                echo '<h6 style="margin-top: 0; color: #dc2626;"><i class="fas fa-code"></i> Output Encoding</h6>';
                echo '<ul style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li>Escape HTML output</li>';
                echo '<li>Sanitize JavaScript strings</li>';
                echo '<li>Use htmlspecialchars()</li>';
                echo '<li>Prevent XSS attacks</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '</div>';
                
                echo '<div style="background: linear-gradient(135deg, #dc2626, #b91c1c); color: white; padding: 25px; border-radius: 12px; margin-top: 30px;">';
                echo '<h6 style="margin-top: 0; color: white;"><i class="fas fa-ban"></i> Critical: Never Do This</h6>';
                echo '<ul style="margin: 0; padding-left: 20px; line-height: 1.8;">';
                echo '<li><strong>Never use eval() or exec()</strong> with user input</li>';
                echo '<li><strong>Never include files based on user input</strong> without validation</li>';
                echo '<li><strong>Never trust data from external sources</strong> without validation</li>';
                echo '<li><strong>Never disable error reporting</strong> in production without logging</li>';
                echo '<li><strong>Never store API keys</strong> in code - use settings</li>';
                echo '</ul>';
                echo '</div>';
                echo '</div>';
                
                // Best Practices
                echo '<div id="best-practices" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-star"></i> Best Practices</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">Follow these best practices for professional, maintainable mod development:</p>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border-left: 4px solid var(--primary);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-code-branch"></i> Code Organization</h6>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Use descriptive function names with mod prefix (e.g., <code>mymod_process_data</code>)</li>';
                echo '<li>Group related functionality together</li>';
                echo '<li>Keep functions small and focused (single responsibility)</li>';
                echo '<li>Add PHPDoc comments for all functions</li>';
                echo '<li>Use consistent naming conventions</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border-left: 4px solid var(--primary);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-bug"></i> Error Handling</h6>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Always include try-catch blocks for external operations</li>';
                echo '<li>Log errors with descriptive messages</li>';
                echo '<li>Provide fallback behavior for failures</li>';
                echo '<li>Never expose error details to end users</li>';
                echo '<li>Use appropriate error levels (info, warning, error)</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border-left: 4px solid var(--primary);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-tachometer-alt"></i> Performance</h6>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Cache expensive operations</li>';
                echo '<li>Minimize database queries</li>';
                echo '<li>Use appropriate hook priorities</li>';
                echo '<li>Avoid processing in loops when possible</li>';
                echo '<li>Profile your mod for bottlenecks</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border-left: 4px solid var(--primary);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-file-alt"></i> Documentation</h6>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Document all public functions</li>';
                echo '<li>Include usage examples</li>';
                echo '<li>Explain complex logic with comments</li>';
                echo '<li>Maintain a changelog</li>';
                echo '<li>Provide installation instructions</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0; border-left: 4px solid var(--primary);">';
                echo '<h6 style="margin-top: 0;"><i class="fas fa-check-circle"></i> Testing</h6>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Test with various data scenarios</li>';
                echo '<li>Verify compatibility with other mods</li>';
                echo '<li>Test error conditions and edge cases</li>';
                echo '<li>Monitor mod logs during testing</li>';
                echo '<li>Test performance impact</li>';
                echo '</ul>';
                echo '</div>';
                echo '</div>';
                
                // Troubleshooting
                echo '<div id="troubleshooting" style="margin-bottom: 50px;">';
                echo '<h5><i class="fas fa-wrench"></i> Troubleshooting</h5>';
                echo '<p style="font-size: 1.05rem; line-height: 1.7;">Common issues and solutions for mod development:</p>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0;">';
                echo '<h6 style="color: var(--primary);"><i class="fas fa-question-circle"></i> Mod Not Loading</h6>';
                echo '<p><strong>Symptoms:</strong> Mod doesn\'t appear in Console → Mod Log</p>';
                echo '<p><strong>Solutions:</strong></p>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Verify file is in <code>/mods/</code> directory</li>';
                echo '<li>Check file has <code>.php</code> extension</li>';
                echo '<li>Ensure init function is called at end of file</li>';
                echo '<li>Check for PHP syntax errors in error log</li>';
                echo '<li>Verify file permissions (644 or 755)</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0;">';
                echo '<h6 style="color: var(--primary);"><i class="fas fa-question-circle"></i> Hook Not Executing</h6>';
                echo '<p><strong>Symptoms:</strong> Hook registered but callback never runs</p>';
                echo '<p><strong>Solutions:</strong></p>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Verify hook name is spelled correctly</li>';
                echo '<li>Check callback function exists and is callable</li>';
                echo '<li>Ensure hook is registered before it\'s executed</li>';
                echo '<li>Test with lower priority number (execute earlier)</li>';
                echo '<li>Add logging to verify registration: <code>SmallMartModAPI::log(\'Hook registered\')</code></li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0;">';
                echo '<h6 style="color: var(--primary);"><i class="fas fa-question-circle"></i> Settings Not Persisting</h6>';
                echo '<p><strong>Symptoms:</strong> Settings reset after page reload</p>';
                echo '<p><strong>Solutions:</strong></p>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Use <code>SmallMartModAPI::setSetting()</code> not PHP variables</li>';
                echo '<li>Verify data directory is writable</li>';
                echo '<li>Check for errors in Console → Error Log</li>';
                echo '<li>Don\'t set settings on every page load - only in init</li>';
                echo '<li>Use unique setting keys to avoid conflicts</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: var(--card); padding: 25px; border-radius: 12px; margin: 20px 0;">';
                echo '<h6 style="color: var(--primary);"><i class="fas fa-question-circle"></i> Performance Issues</h6>';
                echo '<p><strong>Symptoms:</strong> Slow page loads after enabling mod</p>';
                echo '<p><strong>Solutions:</strong></p>';
                echo '<ul style="line-height: 1.8;">';
                echo '<li>Profile code to identify bottlenecks</li>';
                echo '<li>Cache expensive operations (API calls, calculations)</li>';
                echo '<li>Use appropriate hook with right priority</li>';
                echo '<li>Avoid processing in tight loops</li>';
                echo '<li>Consider async processing for heavy tasks</li>';
                echo '</ul>';
                echo '</div>';
                
                echo '<div style="background: linear-gradient(135deg, var(--accent), var(--secondary)); padding: 25px; border-radius: 12px; color: white;">';
                echo '<h6 style="margin-top: 0; color: white;"><i class="fas fa-life-ring"></i> Need Help?</h6>';
                echo '<p style="margin: 0; line-height: 1.7;">If you\'re still experiencing issues:</p>';
                echo '<ol style="line-height: 1.8; margin-top: 10px;">';
                echo '<li>Check Console → Mod Log for your mod\'s messages</li>';
                echo '<li>Review Console → Error Log for PHP errors</li>';
                echo '<li>Enable debug logging: <code>SmallMartModAPI::log($var, \'debug\')</code></li>';
                echo '<li>Test with only your mod enabled</li>';
                echo '<li>Verify SmallMart version compatibility</li>';
                echo '</ol>';
                echo '</div>';
                echo '</div>';
                
                echo '</div>';
                break;
                
            default: // builtin
                echo '<h4><i class="fas fa-puzzle-piece"></i> Built-in  Mods</h4>';
                echo '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(400px, 1fr)); gap: 20px; margin-top: 20px;">';
                
                foreach ($mods as $modId => $mod) {
                    $isEnabled = $mod['enabled'] ?? false;
                    echo '<div class="card" style="padding: 25px; ' . ($isEnabled ? 'border-color: var(--primary);' : '') . '">';
                    
                    echo '<div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 15px;">';
                    echo '<div>';
                    echo '<h5 style="margin: 0 0 8px 0;">' . $mod['name'] . '</h5>';
                    echo '<div style="font-size: 0.8rem; color: var(--text-secondary);">v' . $mod['version'] . ' by ' . $mod['author'] . '</div>';
                    echo '</div>';
                    echo '<div>';
                    if ($isEnabled) {
                        echo '<span class="btn btn-sm" style="background: var(--primary); color: white;">Active</span>';
                    } else {
                        echo '<span class="btn btn-sm btn-secondary">Inactive</span>';
                    }
                    echo '</div>';
                    echo '</div>';
                    
                    echo '<p style="color: var(--text-secondary); margin-bottom: 20px; line-height: 1.5;">' . $mod['description'] . '</p>';
                    
                    if (isset($mod['hooks']) && is_array($mod['hooks'])) {
                        echo '<div style="margin-bottom: 15px;">';
                        echo '<strong style="font-size: 0.8rem; color: var(--text-secondary);">HOOKS:</strong> ';
                        echo '<span style="font-size: 0.8rem;">' . implode(', ', $mod['hooks']) . '</span>';
                        echo '</div>';
                    }
                    
                    echo '<div style="display: flex; gap: 10px; align-items: center;">';
                    echo '<a href="?action=toggle_mod&id=' . $modId . '" class="btn btn-sm ' . ($isEnabled ? 'btn-warning' : 'btn-success') . '">';
                    echo '<i class="fas fa-power-off"></i> ' . ($isEnabled ? 'Disable' : 'Enable');
                    echo '</a>';
                    
                    if (isset($mod['settings']) && !empty($mod['settings'])) {
                        echo '<a href="?admin=mods&edit=' . $modId . '" class="btn btn-sm btn-info"><i class="fas fa-cog"></i> Settings</a>';
                    }
                    
                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';
                break;
        }
    }
    break;
                        
                    default:
                        echo '<div style="text-align: center; padding: 60px;">';
                        echo '<i class="fas fa-tachometer-alt" style="font-size: 4rem; color: var(--primary); margin-bottom: 20px;"></i>';
                        echo '<h3>SmallMart v6 Admin Panel</h3>';
                        echo '<p style="color: var(--text-secondary); margin: 20px 0;">Enhanced Commercial E-Commerce Management System with Console Monitoring</p>';
                        echo '<div style="margin-top: 30px;">';
                        echo '<p><strong>Version:</strong> ' . VERSION . '</p>';
                        echo '<p><strong>Total Items:</strong> ' . count($items) . '</p>';
                        echo '<p><strong>Built-in Mods:</strong> ' . count($builtinMods) . ' available (' . count(array_filter($mods, function($m) { return $m['enabled'] ?? false; })) . ' active)</p>';
                        echo '<p><strong>Built-in Themes:</strong> ' . count($defaultThemes) . ' available</p>';
                        echo '<p><strong>Custom Mods:</strong> ' . count(glob(MODS_DIR . '*.php')) . ' installed</p>';
                        echo '<p><strong>PayPal Integration:</strong> ' . ((!empty($settings['mods']['paypal_integration']['settings']['paypal_email'])) ? 'Connected' : 'Not Configured') . '</p>';
                        echo '<p><strong>Theme System:</strong> Enhanced with Export/Import</p>';
                        echo '<p><strong>Console System:</strong> Real-time monitoring active</p>';
                        echo '<p><strong>Backlink Status:</strong> ' . (isDonationComplete() ? 'Removed (Thank you!)' : 'Active') . '</p>';
                        echo '<p><strong>Copyright:</strong> &copy; ' . COPYRIGHT_YEAR . ' SmallMart</p>';
                        echo '</div>';
                        echo '</div>';
                        break;
                }
                ?>
            </div>
            <?php
        } elseif ($currentItem) {
            // Single item view
            ?>
            <div class="item-view-container fade-in">
                <div>
                    <div class="image-gallery">
                        <?php if (!empty($currentItem['images'])): ?>
                            <div class="main-image" id="main-image">
                                <img src="<?php echo UPLOADS_WEB_PATH . $currentItem['images'][0]; ?>" alt="<?php echo sanitizeInput($currentItem['title'], false); ?>" id="main-image-img">
                            </div>
                            
                            <?php if (count($currentItem['images']) > 1): ?>
                                <div class="thumbnails">
                                    <?php foreach ($currentItem['images'] as $index => $image): ?>
                                        <div class="thumbnail <?php echo $index === 0 ? 'active' : ''; ?>" onclick="changeMainImage('<?php echo UPLOADS_WEB_PATH . $image; ?>', this)">
                                            <img src="<?php echo UPLOADS_WEB_PATH . $image; ?>" alt="Thumbnail <?php echo $index + 1; ?>">
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="main-image">
                                <div style="width: 100%; height: 100%; background: var(--surface); display: flex; align-items: center; justify-content: center; color: var(--text-secondary);">
                                    <i class="fas fa-image" style="font-size: 4rem;"></i>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div>
                    <?php if ($currentItem['featured'] ?? false): ?>
                        <div style="background: linear-gradient(135deg, #fbbf24, #f59e0b); color: white; padding: 8px 16px; border-radius: 20px; display: inline-block; margin-bottom: 15px; font-weight: 600;">
                            <i class="fas fa-star"></i> Featured Item
                        </div>
                    <?php endif; ?>
                    
                    <h1 style="font-size: 2.2rem; margin-bottom: 15px;"><?php echo sanitizeInput($currentItem['title'], false); ?></h1>
                    
                    <span class="item-type <?php echo $currentItem['type']; ?>" style="font-size: 1rem; padding: 8px 20px; margin-bottom: 20px; display: inline-block;">
                        <?php echo ucfirst(str_replace('_', ' ', $currentItem['type'])); ?>
                    </span>
                    
                    <div class="item-price" style="font-size: 2.5rem; margin-bottom: 20px;">
                        <?php echo formatPrice($currentItem['price'], $settings['currency']); ?>
                    </div>
                    
                    <div class="item-meta" style="border-top: 1px solid var(--border); border-bottom: 1px solid var(--border); padding: 15px 0; margin: 20px 0;">
                        <div><strong>Condition:</strong> <?php echo ucfirst($currentItem['condition'] ?? 'new'); ?></div>
                        <?php if (!empty($currentItem['location'])): ?>
                            <div><strong>Location:</strong> <?php echo sanitizeInput($currentItem['location'], false); ?></div>
                        <?php endif; ?>
                        <div><strong>Views:</strong> <?php echo $currentItem['views'] ?? 0; ?></div>
                        <div><strong>Stock:</strong> <?php echo $currentItem['stock'] ?? 1; ?></div>
                        <?php if (!empty($currentItem['tags'])): ?>
                            <div><strong>Tags:</strong> <?php echo sanitizeInput($currentItem['tags'], false); ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div style="margin: 25px 0;">
                        <h3><i class="fas fa-info-circle"></i> Description</h3>
                        <div style="line-height: 1.7; color: var(--text-secondary);">
                            <?php echo $currentItem['description']; ?>
                        </div>
                    </div>
                    
                    <!-- Payment Options -->
                    <div class="card" style="padding: 25px; margin: 20px 0;">
                        <h3><i class="fas fa-shopping-cart"></i> Get This Item</h3>
                        
                        <?php if (!empty($settings['require_login_for_purchase']) && !isUserLoggedIn()): ?>
                            <div class="message error fade-in" style="margin: 15px 0;">
                                <i class="fas fa-user-lock"></i>
                                Please sign in (or create an account) to purchase or contact the seller.
                            </div>
                            <div style="display:flex; gap:10px; flex-wrap:wrap; margin-top:10px;">
                                <a class="btn btn-secondary btn-lg" href="?account=login">
                                    <i class="fas fa-sign-in-alt"></i> Login
                                </a>
                                <a class="btn btn-secondary btn-lg" href="?account=register">
                                    <i class="fas fa-user-plus"></i> Sign Up
                                </a>
                            </div>
                        <?php else: ?>
                        <?php if (($currentItem['stock'] ?? 1) > 0 && is_numeric($currentItem['price']) && $currentItem['price'] > 0): ?>
                            <div style="margin: 20px 0;">
                                <?php $paypalForm = generatePayPalForm($currentItem, $settings); ?>
                                <?php if ($paypalForm): ?>
                                    <?php echo $paypalForm; ?>
                                <?php elseif ($settings['payment_methods']['paypal'] && ($currentItem['enable_paypal'] ?? true)): ?>
                                    <div style="background: rgba(239, 68, 68, 0.1); color: #dc2626; padding: 15px; border-radius: 8px; margin: 10px 0;">
                                        <i class="fas fa-exclamation-triangle"></i> PayPal not configured. Please add your PayPal email in admin settings.
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (isContactSellerEnabled($currentItem, $settings)): ?>
                                    <a href="mailto:<?php echo sanitizeInput($settings['contact_email'], false); ?>?subject=Inquiry about <?php echo urlencode($currentItem['title']); ?>" class="btn btn-secondary btn-lg" style="margin: 5px 0; width: 100%;">
                                        <i class="fas fa-envelope"></i> Contact Seller
                                    </a>
                                <?php endif; ?>
                                
                                <?php if (!$paypalForm && !isContactSellerEnabled($currentItem, $settings)): ?>
                                    <div style="text-align: center; padding: 20px; background: rgba(255, 165, 0, 0.1); border-radius: 8px; color: #ff8c00;">
                                        <i class="fas fa-info-circle"></i> No payment methods enabled for this item.
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <div style="text-align: center; padding: 30px; background: rgba(239, 68, 68, 0.1); border-radius: 12px; color: #dc2626;">
                                <h4>Not Available for Purchase</h4>
                                <p>This item is currently unavailable or contact seller for pricing.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                                        <?php endif; ?>

                    <?php if ($settings['enable_wishlist'] ?? true): ?>
                        <button class="btn btn-secondary" style="width: 100%; margin-top: 15px;" onclick="addToWishlist('<?php echo $currentItem['id']; ?>')">
                            <i class="fas fa-heart"></i> Add to Wishlist
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Social Sharing -->
            <div class="card fade-in" style="padding: 25px; margin: 30px 0;">
                <h3><i class="fas fa-share-alt"></i> Share This Item</h3>
                <div style="display: flex; gap: 10px; flex-wrap: wrap; margin-top: 15px;">
                    <a href="https://www.facebook.com/sharer/sharer.php?u=<?php echo urlencode('http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']); ?>" target="_blank" class="btn btn-sm" style="background: #1877f2;">
                        <i class="fab fa-facebook-f"></i> Facebook
                    </a>
                    <a href="https://twitter.com/intent/tweet?url=<?php echo urlencode('http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']); ?>&text=<?php echo urlencode($currentItem['title']); ?>" target="_blank" class="btn btn-sm" style="background: #1da1f2;">
                        <i class="fab fa-twitter"></i> Twitter
                    </a>
                    <a href="https://wa.me/?text=<?php echo urlencode($currentItem['title'] . ' ' . 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']); ?>" target="_blank" class="btn btn-sm" style="background: #25d366;">
                        <i class="fab fa-whatsapp"></i> WhatsApp
                    </a>
                    <a href="https://pinterest.com/pin/create/button/?url=<?php echo urlencode('http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']); ?>&description=<?php echo urlencode($currentItem['title']); ?>" target="_blank" class="btn btn-sm" style="background: #bd081c;">
                        <i class="fab fa-pinterest"></i> Pinterest
                    </a>
                </div>
            </div>
            <?php
        } else {
            // Main storefront display
            ?>
            <div class="fade-in">
                <?php if ($search): ?>
                    <div style="margin: 20px 0; padding: 20px; background: var(--surface); border-radius: 12px;">
                        <h3><i class="fas fa-search"></i> Search Results for "<?php echo sanitizeInput($search, false); ?>"</h3>
                        <p style="color: var(--text-secondary);">Found <?php echo count($filteredItems); ?> items</p>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($paginatedItems)): ?>
                    <div class="items-grid">
                        <?php foreach ($paginatedItems as $item): ?>
                            <div class="item-card card">
                                <?php if ($item['featured'] ?? false): ?>
                                    <div class="item-featured">Featured</div>
                                <?php endif; ?>
                                
                                <div class="item-image">
                                    <?php if (!empty($item['images'])): ?>
                                        <img src="<?php echo UPLOADS_WEB_PATH . $item['images'][0]; ?>" alt="<?php echo sanitizeInput($item['title'], false); ?>">
                                        <?php if (count($item['images']) > 1): ?>
                                            <div style="position: absolute; bottom: 10px; right: 10px; background: rgba(0,0,0,0.8); color: white; padding: 4px 8px; border-radius: 12px; font-size: 12px;">
                                                <i class="fas fa-images"></i> <?php echo count($item['images']); ?>
                                            </div>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <div style="width: 100%; height: 100%; background: var(--surface); display: flex; align-items: center; justify-content: center; color: var(--text-secondary);">
                                            <i class="fas fa-image" style="font-size: 3rem;"></i>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="item-details">
                                    <h3 class="item-title">
                                        <a href="?view=<?php echo $item['id']; ?>" style="color: var(--text); text-decoration: none;">
                                            <?php echo sanitizeInput($item['title'], false); ?>
                                        </a>
                                    </h3>
                                    
                                    <div class="item-description">
                                        <?php echo sanitizeInput(substr(strip_tags($item['description']), 0, 150), false) . (strlen(strip_tags($item['description'])) > 150 ? '...' : ''); ?>
                                    </div>
                                    
                                    <div class="item-price">
                                        <?php echo formatPrice($item['price'], $settings['currency']); ?>
                                    </div>
                                    
                                    <span class="item-type <?php echo $item['type']; ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $item['type'])); ?>
                                    </span>
                                    
                                    <div class="item-meta">
                                        <span><i class="fas fa-eye"></i> <?php echo $item['views'] ?? 0; ?> views</span>
                                        <?php if (!empty($item['location'])): ?>
                                            <span><i class="fas fa-map-marker-alt"></i> <?php echo sanitizeInput($item['location'], false); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div style="display: flex; gap: 10px; margin-top: 15px;">
                                        <a href="?view=<?php echo $item['id']; ?>" class="btn btn-sm" style="flex: 1;">View Details</a>
                                        <?php if ($settings['enable_wishlist'] ?? true): ?>
                                            <button class="btn btn-sm btn-secondary" title="Add to Wishlist" onclick="addToWishlist('<?php echo $item['id']; ?>')">
                                                <i class="fas fa-heart"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <?php if ($page > 1): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                    <i class="fas fa-chevron-left"></i> Previous
                                </a>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <?php if ($i == $page): ?>
                                    <span class="current"><?php echo $i; ?></span>
                                <?php else: ?>
                                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                <?php endif; ?>
                            <?php endfor; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                    Next <i class="fas fa-chevron-right"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div style="text-align: center; padding: 80px 20px; background: var(--surface); border-radius: 12px;">
                        <i class="fas fa-search" style="font-size: 4rem; color: var(--text-secondary); margin-bottom: 20px;"></i>
                        <h3>No Items Found</h3>
                        <?php if ($search): ?>
                            <p style="color: var(--text-secondary); margin: 15px 0;">No items match your search criteria.</p>
                            <a href="?" class="btn">Browse All Items</a>
                        <?php else: ?>
                            <p style="color: var(--text-secondary); margin: 15px 0;">No items are currently available.</p>
                            <?php if (isAdmin()): ?>
                                <a href="?admin=add_item" class="btn">Add Your First Item</a>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            <?php
        }
        ?>
    </div>
    
    <!-- Footer -->
    <footer>
        <div class="container">
            <div class="footer-content">
                <div style="text-align: center;">
                    <?php echo $settings['footer_text']; ?>
                </div>
                
                <div class="footer-links">
                    <a href="?">Home</a>
                    <a href="?mode=all">All Items</a>
                    <?php if ($settings['enabled_modes']['auction'] ?? false): ?>
                        <a href="?mode=auction">Auctions</a>
                    <?php endif; ?>
                    <?php if ($settings['enabled_modes']['buy_now'] ?? false): ?>
                        <a href="?mode=buy_now">Buy Now</a>
                    <?php endif; ?>
                    <?php if ($settings['enabled_modes']['make_offer'] ?? false): ?>
                        <a href="?mode=make_offer">Make Offer</a>
                    <?php endif; ?>
                    <?php if ($settings['enable_wishlist'] ?? true): ?>
                        <a href="?wishlist=view">Wishlist</a>
                    <?php endif; ?>
                    
                    <?php $footerPages = loadPages(); ?>
                    <?php foreach ($footerPages as $pslug => $pg): ?>
                        <?php if (!empty($pg['show_in_footer'])): ?>
                            <a href="?p=<?php echo urlencode((string)$pslug); ?>"><?php echo sanitizeInput((string)($pg['title'] ?? $pslug), false); ?></a>
                        <?php endif; ?>
                    <?php endforeach; ?>
<?php if (isAdmin()): ?>
                        <a href="?admin=dashboard">Admin</a>
                    <?php endif; ?>
                </div>
                
                <div style="margin-top: 20px; font-size: 0.9rem; opacity: 0.8;">
                    <?php echo $settings['powered_by']; ?>
                </div>
                
                <!-- Conditional Backlink -->
                <?php if (shouldShowBacklink()): ?>
                <div style="margin-top: 15px;">
                    <a href="https://smallmart.site" target="_blank" class="backlink">
                        <i class="fas fa-external-link-alt"></i>
                        SmallMart -  E-Commerce Solution
                    </a>
                </div>
                <?php endif; ?>
            </div>
        </div>
                <?php SmallMartModAPI::executeHook('footer_after'); ?>
        </footer>
    
    <script>
        // Global variables
        let quillEditor = null;
        
        // Helper function for formatBytes
        function formatBytes(bytes, decimals = 2) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const dm = decimals < 0 ? 0 : decimals;
            const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
        }
        
        // Initialize application
        document.addEventListener('DOMContentLoaded', function() {
            console.log('SmallMart v6 Enhanced Mod System initialized successfully');
            
            // Initialize WYSIWYG editor if enabled
            <?php if ($settings['enable_wysiwyg'] ?? true): ?>
            if (document.getElementById('description-editor')) {
                initializeWYSIWYG();
            }
            <?php endif; ?>
            
            // Initialize form handlers
            initializeFormHandlers();
            
            // Initialize theme builder if on that page
            if (document.getElementById('theme-builder-form')) {
                initializeThemeBuilder();
            }
            
            // Initialize export theme selector
            const exportSelect = document.getElementById('export-theme-select');
            const exportBtn = document.getElementById('export-btn');
            if (exportSelect && exportBtn) {
                exportSelect.addEventListener('change', function() {
                    const themeId = this.value;
                    exportBtn.href = '?action=export_theme&id=' + encodeURIComponent(themeId);
                });
            }
            
            // Add theme change confirmation
            document.querySelectorAll('form[action*="save_settings"]').forEach(form => {
                if (form.querySelector('input[name="change_theme_only"]')) {
                    form.addEventListener('submit', function(e) {
                        const themeName = form.querySelector('select[name="theme"] option:checked').text;
                        if (!confirm('Switch to theme: ' + themeName + '? The page will refresh with the new theme.')) {
                            e.preventDefault();
                        }
                    });
                }
            });
            
            // Console auto-refresh for live monitoring
            if (window.location.href.includes('admin=console')) {
                setInterval(function() {
                    const liveStats = document.getElementById('live-stats');
                    if (liveStats) {
                        // Update memory usage
                        const memoryElement = document.getElementById('memory-usage');
                        if (memoryElement) {
                            fetch('?admin=console&ajax=memory')
                                .then(response => response.text())
                                .catch(() => {}); // Silent fail for live updates
                        }
                    }
                }, 30000); // Update every 30 seconds
            }
            
            // Debug theme loading
            const currentTheme = '<?php echo $settings['theme'] ?? 'unknown'; ?>';
            console.log('Current theme loaded: ' + currentTheme);
            console.log('Mod system: Enhanced with custom mod upload capability');
            console.log('Theme system: Enhanced with export/import functionality');
            console.log('Built-in Mods: <?php echo count($builtinMods); ?> available');
            console.log('Built-in Themes: <?php echo count($defaultThemes); ?> available');
            console.log('Console System: Real-time monitoring active');
            
            // Auto-hide messages
            setTimeout(() => {
                document.querySelectorAll('.message').forEach(msg => {
                    msg.style.opacity = '0';
                    setTimeout(() => msg.remove(), 300);
                });
            }, 5000);
        });
        
        // WYSIWYG Editor
        function initializeWYSIWYG() {
            if (typeof Quill !== 'undefined') {
                quillEditor = new Quill('#description-editor', {
                    theme: 'snow',
                    modules: {
                        toolbar: [
                            ['bold', 'italic', 'underline', 'strike'],
                            ['blockquote', 'code-block'],
                            [{ 'header': 1 }, { 'header': 2 }],
                            [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                            [{ 'script': 'sub'}, { 'script': 'super' }],
                            [{ 'indent': '-1'}, { 'indent': '+1' }],
                            [{ 'size': ['small', false, 'large', 'huge'] }],
                            [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
                            [{ 'color': [] }, { 'background': [] }],
                            [{ 'align': [] }],
                            ['link', 'image'],
                            ['clean']
                        ]
                    },
                    placeholder: 'Enter a detailed description...'
                });
                
                const textarea = document.getElementById('description-textarea');
                if (textarea && textarea.value) {
                    quillEditor.root.innerHTML = textarea.value;
                }
                
                quillEditor.on('text-change', function() {
                    textarea.value = quillEditor.root.innerHTML;
                });
            }
        }
        
        // Form handlers
        function initializeFormHandlers() {
            const typeSelect = document.querySelector('select[name="type"]');
            if (typeSelect) {
                typeSelect.addEventListener('change', toggleItemTypeFields);
                toggleItemTypeFields();
            }
            
            const itemForm = document.querySelector('form[action*="save_item"]');
            if (itemForm && quillEditor) {
                itemForm.addEventListener('submit', function() {
                    const textarea = document.getElementById('description-textarea');
                    if (textarea && quillEditor) {
                        textarea.value = quillEditor.root.innerHTML;
                    }
                });
            }
        }
        
        // Theme Builder Functions
        function initializeThemeBuilder() {
            document.querySelectorAll('.color-input').forEach(input => {
                input.addEventListener('input', updatePreview);
            });
            
            document.querySelectorAll('select[name^="font_"]').forEach(select => {
                select.addEventListener('change', updatePreview);
            });
            
            updatePreview();
        }
        
        function updateColorFromPicker(picker, fieldName) {
            const textInput = document.querySelector(`input[name="${fieldName}"]`);
            if (textInput) {
                textInput.value = picker.value;
                updatePreview();
            }
        }
        
        function updatePreview() {
            const preview = document.getElementById('theme-preview');
            if (!preview) return;
            
            const colors = {
                primary: document.querySelector('input[name="primary"]')?.value || '#ffcc00',
                secondary: document.querySelector('input[name="secondary"]')?.value || '#ff9900',
                accent: document.querySelector('input[name="accent"]')?.value || '#ffd700',
                background: document.querySelector('input[name="background"]')?.value || '#1a1a1a',
                surface: document.querySelector('input[name="surface"]')?.value || '#2d2d2d',
                card: document.querySelector('input[name="card"]')?.value || '#3c3c3c',
                text: document.querySelector('input[name="text"]')?.value || '#ffffff',
                text_secondary: document.querySelector('input[name="text_secondary"]')?.value || '#e0e0e0',
                border: document.querySelector('input[name="border"]')?.value || '#ffcc0033'
            };
            
            const fonts = {
                primary: document.querySelector('select[name="font_primary"]')?.value || "'Inter', sans-serif",
                heading: document.querySelector('select[name="font_heading"]')?.value || "'Poppins', sans-serif"
            };
            
            const cssVars = `
                --primary: ${colors.primary};
                --secondary: ${colors.secondary};
                --accent: ${colors.accent};
                --background: ${colors.background};
                --surface: ${colors.surface};
                --card: ${colors.card};
                --text: ${colors.text};
                --text-secondary: ${colors.text_secondary};
                --border: ${colors.border};
                --font-primary: ${fonts.primary};
                --font-heading: ${fonts.heading};
            `;
            
            preview.style.cssText = cssVars;
            
            Object.keys(colors).forEach(colorKey => {
                const colorValue = colors[colorKey];
                const colorPicker = document.querySelector(`input[type="color"][onchange*="${colorKey}"]`);
                if (colorPicker && colorValue.match(/^#[0-9A-Fa-f]{6}$/)) {
                    colorPicker.value = colorValue;
                }
            });
        }
        
        // Toggle item type specific fields
        function toggleItemTypeFields() {
            const typeSelect = document.querySelector('select[name="type"]');
            const auctionFields = document.getElementById('auction-fields');
            const offerFields = document.getElementById('offer-fields');
            
            if (!typeSelect) return;
            
            const selectedType = typeSelect.value;
            
            if (auctionFields) {
                auctionFields.style.display = selectedType === 'auction' ? 'block' : 'none';
            }
            
            if (offerFields) {
                offerFields.style.display = selectedType === 'make_offer' ? 'block' : 'none';
            }
        }
        
        // Handle right-click for admin access
        function handleRightClick(event) {
            event.preventDefault();
            window.location.href = '?admin_login=1';
            return false;
        }
        // Image gallery functions
        function changeMainImage(imageSrc, thumbnail) {
            const mainImg = document.getElementById('main-image-img');
            if (mainImg) {
                mainImg.src = imageSrc;
            }
            
            // Update active thumbnail
            document.querySelectorAll('.thumbnail').forEach(thumb => {
                thumb.classList.remove('active');
            });
            if (thumbnail) {
                thumbnail.classList.add('active');
            }
        }
        
        // Wishlist functions
        function addToWishlist(itemId) {
            let wishlist = JSON.parse(localStorage.getItem('smallmart_wishlist') || '[]');
            
            if (!wishlist.includes(itemId)) {
                wishlist.push(itemId);
                localStorage.setItem('smallmart_wishlist', JSON.stringify(wishlist));
                showNotification('Item added to wishlist!', 'success');
            } else {
                showNotification('Item already in wishlist', 'info');
            }
        }
        
        function removeFromWishlist(itemId) {
            let wishlist = JSON.parse(localStorage.getItem('smallmart_wishlist') || '[]');
            wishlist = wishlist.filter(id => id !== itemId);
            localStorage.setItem('smallmart_wishlist', JSON.stringify(wishlist));
            showNotification('Item removed from wishlist', 'success');
            
            // Refresh wishlist view if we're on that page
            if (window.location.href.includes('wishlist=view')) {
                location.reload();
            }
        }
        
        // Notification system
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `message ${type} fade-in`;
            notification.innerHTML = `<i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'}"></i> ${message}`;
            notification.style.position = 'fixed';
            notification.style.top = '20px';
            notification.style.right = '20px';
            notification.style.zIndex = '9999';
            notification.style.maxWidth = '300px';
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                setTimeout(() => notification.remove(), 300);
            }, 3000);
        }
        
        // Console functions
        function refreshSystemInfo() {
            showNotification('Refreshing system information...', 'info');
            setTimeout(() => {
                location.reload();
            }, 1000);
        }
        
        function showPHPInfo() {
            const popup = window.open('', 'phpinfo', 'width=800,height=600,scrollbars=yes');
            popup.document.write('<h1>PHP Information</h1><p>PHP Version: <?php echo phpversion(); ?></p><p>Server Software: <?php echo $_SERVER["SERVER_SOFTWARE"] ?? "Unknown"; ?></p>');
        }
        
        // Advanced search functions
        function toggleAdvancedSearch() {
            const advancedPanel = document.getElementById('advanced-search-panel');
            if (advancedPanel) {
                advancedPanel.style.display = advancedPanel.style.display === 'none' ? 'block' : 'none';
            }
        }
        
        // Theme preview functions
        function previewTheme(themeId) {
            const themes = <?php echo json_encode($themes); ?>;
            if (themes[themeId]) {
                const theme = themes[themeId];
                if (theme.colors) {
                    Object.keys(theme.colors).forEach(colorKey => {
                        document.documentElement.style.setProperty('--' + colorKey.replace('_', '-'), theme.colors[colorKey]);
                    });
                }
            }
        }
        
        // Form validation
        function validateItemForm() {
            const title = document.querySelector('input[name="title"]').value.trim();
            const price = document.querySelector('input[name="price"]').value.trim();
            
            if (!title) {
                showNotification('Title is required', 'error');
                return false;
            }
            
            if (!price) {
                showNotification('Price is required', 'error');
                return false;
            }
            
            return true;
        }
        
        // Price formatting
        function formatPriceInput(input) {
            let value = input.value.replace(/[^\d.-]/g, '');
            if (value && !isNaN(value)) {
                value = parseFloat(value).toFixed(2);
                input.value = value;
            }
        }
        
        // Category management
        function toggleCategory(categoryId) {
            const currentUrl = new URL(window.location);
            if (currentUrl.searchParams.get('category') === categoryId) {
                currentUrl.searchParams.delete('category');
            } else {
                currentUrl.searchParams.set('category', categoryId);
            }
            window.location.href = currentUrl.toString();
        }
        
        // Lazy loading for images
        function initializeLazyLoading() {
            const images = document.querySelectorAll('img[data-src]');
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        observer.unobserve(img);
                    }
                });
            });
            
            images.forEach(img => imageObserver.observe(img));
        }
        
        // Search suggestions
        function initializeSearchSuggestions() {
            const searchInput = document.querySelector('.search-input');
            if (searchInput) {
                let timeout;
                searchInput.addEventListener('input', function() {
                    clearTimeout(timeout);
                    timeout = setTimeout(() => {
                        const query = this.value.trim();
                        if (query.length > 2) {
                            // Implement search suggestions here
                            console.log('Search suggestions for:', query);
                        }
                    }, 300);
                });
            }
        }
        
        // Analytics tracking
        function trackEvent(eventType, itemId = null) {
            // Simple analytics tracking
            console.log('Event tracked:', eventType, itemId);
        }
        
        // Mobile menu toggle
        function toggleMobileMenu() {
            const mobileMenu = document.getElementById('mobile-menu');
            if (mobileMenu) {
                mobileMenu.classList.toggle('active');
            }
        }
        
        // Initialize additional features
        document.addEventListener('DOMContentLoaded', function() {
            initializeLazyLoading();
            initializeSearchSuggestions();
            
            // Track page view
            trackEvent('page_view');
            
            // Initialize tooltips
            document.querySelectorAll('[title]').forEach(element => {
                element.addEventListener('mouseenter', function() {
                    // Simple tooltip implementation
                });
            });
        });
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + K for search
            if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
                e.preventDefault();
                const searchInput = document.querySelector('.search-input');
                if (searchInput) {
                    searchInput.focus();
                }
            }
            
            // Escape to close modals
            if (e.key === 'Escape') {
                document.querySelectorAll('.modal.active').forEach(modal => {
                    modal.classList.remove('active');
                });
            }
        });
        
        // Auto-save functionality for admin forms
        function enableAutoSave(formSelector) {
            const form = document.querySelector(formSelector);
            if (form) {
                const inputs = form.querySelectorAll('input, textarea, select');
                inputs.forEach(input => {
                    input.addEventListener('change', function() {
                        const formData = new FormData(form);
                        localStorage.setItem('smallmart_autosave_' + form.id, JSON.stringify(Object.fromEntries(formData)));
                    });
                });
                
                // Restore auto-saved data
                const savedData = localStorage.getItem('smallmart_autosave_' + form.id);
                if (savedData) {
                    try {
                        const data = JSON.parse(savedData);
                        Object.keys(data).forEach(key => {
                            const input = form.querySelector(`[name="${key}"]`);
                            if (input && input.type !== 'file') {
                                input.value = data[key];
                            }
                        });
                    } catch (e) {
                        console.log('Error restoring auto-saved data:', e);
                    }
                }
            }
        }
        
        // Initialize auto-save for admin forms
        if (window.location.href.includes('admin=')) {
            enableAutoSave('#theme-builder-form');
            enableAutoSave('form[action*="save_item"]');
            enableAutoSave('form[action*="save_settings"]');
        }
        
        // Performance monitoring
        window.addEventListener('load', function() {
            const loadTime = performance.now();
            console.log('SmallMart loaded in:', Math.round(loadTime), 'ms');
            
            // Track performance
            trackEvent('page_load_time', Math.round(loadTime));
        });
        
        // Service worker registration for PWA features
        if ('serviceWorker' in navigator) {
            window.addEventListener('load', function() {
                // Service worker would be registered here for PWA features
                console.log('Service Worker support detected');
            });
        }
    </script>
</body>
</html>

<?php
// Final system cleanup and optimization
if (function_exists('fastcgi_finish_request')) {
    fastcgi_finish_request();
}

// Log successful page load
Logger::system('Page rendered successfully for: ' . ($_SERVER['REQUEST_URI'] ?? '/'));

// Execute any end-of-request hooks
SmallMartModAPI::executeHook('request_end');
?>